import { Formik, FormikErrors, validateYupSchema, yupToFormErrors } from 'formik';
import { FormikConfig, FormikValues } from 'formik/dist/types';
import React, { useCallback } from 'react';

const emptyErrors: FormikErrors<unknown> = {};

interface Props {
   validationContext?: any;
}

export function FixedFormik<Values extends FormikValues = FormikValues, ExtraProps = {}>(
   props: Props & FormikConfig<Values> & ExtraProps,
) {
   // eslint-disable-next-line @typescript-eslint/no-unused-vars
   const { validationContext, validationSchema, validate, ...rest } = props;

   const handleValidate = useCallback(
      (values: Values): Promise<FormikErrors<Values>> => {
         if (!validationSchema) {
            return Promise.resolve(emptyErrors);
         }

         // почти копипаста runValidationSchema из недр Formik
         return new Promise((resolve, reject) => {
            validateYupSchema(values, validationSchema, false, validationContext).then(
               () => resolve(emptyErrors),
               (err: any) => {
                  // Yup will throw a validation error if validation fails. We catch those and
                  // resolve them into Formik errors. We can sniff if something is a Yup error
                  // by checking error.name.
                  // @see https://github.com/jquense/yup#validationerrorerrors-string--arraystring-value-any-path-string
                  if (err.name === 'ValidationError') {
                     resolve(yupToFormErrors(err));
                  } else {
                     // We throw any other errors
                     if (process.env.NODE_ENV !== 'production') {
                        console.warn(
                           `Warning: An unhandled error was caught during validation in <Formik validationSchema />`,
                           err,
                        );
                     }

                     reject(err);
                  }
               },
            );
         });
      },
      [validationContext, validationSchema],
   );

   return <Formik validate={validationSchema ? handleValidate : undefined} {...rest} />;
}

FixedFormik.displayName = 'FixedFormik';
