import { Card } from '@yandex-cloud/uikit';
import React, { ReactNode } from 'react';

import { classNames } from '../../../formatters';
import { RemoveButton } from '../../../small_components';

import classes from './FormCard.module.css';

export interface FormCardProps {
   buttons?: ReactNode;
   className?: string;
   collapseHeader?: boolean;
   hasRemoveButton?: boolean;
   qa?: string;
   subTitle?: ReactNode;
   title: ReactNode;

   onRemove?(): void;
}

export const FormCard: React.FC<FormCardProps> = React.memo(
   ({
      buttons,
      children,
      className,
      collapseHeader = false,
      hasRemoveButton = false,
      onRemove,
      qa,
      subTitle,
      title,
   }) => {
      if (hasRemoveButton && !onRemove) {
         throw new Error('FormCard: expected onRemove, if hasRemoveButton is true');
      }

      return (
         <div data-qa={qa || 'FormCard:Section'}>
            <Card className={classNames(classes.card, className, { [classes.collapse]: collapseHeader })}>
               {(title || buttons || hasRemoveButton) && (
                  <header className={classes.header} data-qa={'FormCard:Header'}>
                     <div>
                        <div className={classes.title} data-qa={'FormCard:Title'}>
                           {title &&
                              (typeof title === 'string' ? <h3 className={classes.simpleTitle}>{title}</h3> : title)}
                        </div>
                        {subTitle && <div className={classes.subTitle}>{subTitle}</div>}
                     </div>

                     {(buttons || hasRemoveButton) && (
                        <div className={classes.buttons} data-qa={'FormCard:Buttons'}>
                           {buttons}

                           {hasRemoveButton && onRemove ? <RemoveButton onClick={onRemove} size={'lg'} /> : null}
                        </div>
                     )}
                  </header>
               )}

               <div className={classes.body} data-qa={'FormCard:Body'}>
                  {children}
               </div>
            </Card>
         </div>
      );
   },
);

FormCard.displayName = 'FormCard';
