import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { bool, object, string } from 'yup';

import { json } from '../../formatters';
import { ExtendedFieldConfig } from '../models';

import { CheckboxField } from './CheckboxField';

const initialValues = {
   empty: null as boolean | null,
   regular: true,
   withError: null as boolean | null,
};

type FormValue = typeof initialValues;

function CheckboxFieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         regular: bool().required(),
         withError: string().required(),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <CheckboxField {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

export const Regular = () => <CheckboxFieldStory name={'regular'}>Regular</CheckboxFieldStory>;

export const Disabled = () => (
   <CheckboxFieldStory name={'regular'} disabled={true}>
      Disabled
   </CheckboxFieldStory>
);

export const Readonly = () => (
   <CheckboxFieldStory name={'regular'} readonly={true}>
      Readonly
   </CheckboxFieldStory>
);

export const Indeterminate = () => <CheckboxFieldStory name={'empty'}>Indeterminate</CheckboxFieldStory>;

export const IndeterminateReadonly = () => (
   <CheckboxFieldStory name={'empty'} readonly={true}>
      IndeterminateReadonly
   </CheckboxFieldStory>
);

export const IndeterminateDisabled = () => (
   <CheckboxFieldStory name={'empty'} controlProps={{ disableIndeterminate: true }}>
      IndeterminateDisabled
   </CheckboxFieldStory>
);

export const WithError = () => (
   <CheckboxFieldStory name={'withError'} label={'With error'}>
      WithError
   </CheckboxFieldStory>
);

export const Playground = () => (
   <CheckboxFieldStory
      name={'regular'}
      label={text('label', 'label')}
      placeholder={text('placeholder', 'placeholder')}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
   >
      {text('children', 'Checkbox label')}
   </CheckboxFieldStory>
);

export default {
   title: 'forms2/fields/CheckboxField',
} as Meta;
