import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { object, string } from 'yup';

import { json } from '../../formatters';
import { ExtendedFieldConfig } from '../models';

import { EnumField } from './EnumField';

const initialValues = {
   empty: '',
   regular: 'xxx',
   withError: '',
};

type FormValue = typeof initialValues;

function EnumFieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: string().required(),
         xField: string().min(10),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <EnumField {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

const controlProps = {
   options: [
      // если мы конвертируем title -> content внутри компонента,
      // то стоит проверить, что мы это делаем валидно
      { value: 'xxx', title: 'XXX' },
      { value: 'yyy', title: 'YYY' },
      { value: 'zzz', title: 'ZZZ' },
   ],
};

export const RegularDeclarative = () => (
   <EnumFieldStory name={'regular'} label={'Regular'} controlProps={controlProps} />
);

export const Disabled = () => (
   <EnumFieldStory name={'regular'} label={'Disabled'} controlProps={controlProps} disabled={true} />
);

export const Readonly = () => (
   <EnumFieldStory name={'regular'} label={'Readonly'} controlProps={controlProps} readonly={true} />
);

export const Empty = () => (
   <EnumFieldStory name={'empty'} label={'Readonly & Empty'} controlProps={controlProps} readonly={true} />
);

export const WithError = () => <EnumFieldStory name={'withError'} label={'With error'} controlProps={controlProps} />;

export const Playground = () => (
   <EnumFieldStory
      name={'regular'}
      label={text('label', 'label')}
      controlProps={controlProps}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
   />
);

export default {
   title: 'forms2/fields/EnumField',
} as Meta;
