import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { object, string } from 'yup';

import { json } from '../../formatters';
import { WarningPlate } from '../../small_components';
import { ExtendedFieldConfig } from '../models';

import { InputField } from './InputField';
import classes from './InputField.stories.module.css';

const initialValues = {
   age: 0,
   empty: '',
   regular: 'initialValue',
   withError: '',
};

type FormValue = typeof initialValues;

function InputFieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: string().required(),
         xField: string().min(10),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <InputField placeholder={'Placeholder'} {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

export const Regular = () => <InputFieldStory name={'regular'} label={'Regular'} />;

export const Disabled = () => <InputFieldStory name={'regular'} label={'Disabled'} disabled={true} />;

export const Readonly = () => <InputFieldStory name={'regular'} label={'Readonly'} readonly={true} />;

export const Empty = () => <InputFieldStory name={'empty'} label={'Readonly & Empty'} readonly={true} />;

export const WithError = () => <InputFieldStory name={'withError'} label={'With error'} />;

export const Playground = () => (
   <InputFieldStory
      disabled={boolean('disabled', false)}
      help={text('help', 'help text')}
      hidden={boolean('hidden', false)}
      hint={text('hint', 'short hint')}
      label={text('label', 'label')}
      name={'regular'}
      placeholder={text('placeholder', 'placeholder')}
      readonly={boolean('readonly', false)}
      required={boolean('required', false)}
   />
);

export const Number = () => <InputFieldStory name={'age'} label={'Age'} controlProps={{ type: 'number' }} />;

export const WithWarningPlate = () => (
   <InputFieldStory
      controlProps={{ type: 'number' }}
      help={(name, form) => {
         const age = form.getFieldProps(name).value;

         return age > 80 ? <WarningPlate className={classes.warning}>{age} is over 80</WarningPlate> : null;
      }}
      label={'Age'}
      name={'age'}
      placeholder={'Enter value over 80'}
   />
);

export default {
   title: 'forms2/fields/InputField',
} as Meta;
