import { TextInput, TextInputProps } from '@yandex-cloud/uikit';
import React, { useCallback } from 'react';

import { EMPTY_VALUE } from '../../_models';
import { styleHelpers } from '../../_styles/styleHelpers';
import { classNames } from '../../formatters';
import { usePreventScrollForInputs } from '../../react_hooks';
import { FieldLayout } from '../components';
import { useExtendedField } from '../helpers';
import { ExtendedFieldConfig } from '../models';

import classes from './InputField.module.css';

interface InputControlProps extends Partial<TextInputProps> {
   /**
    * Тип для компонента <TextInput/>
    */
   type?: 'text' | 'email' | 'tel' | 'password' | 'number';
}

export const InputField = React.memo((props: ExtendedFieldConfig<any, string | number | null, InputControlProps>) => {
   const { field, readonly, disabled, onChange, onBlur, showError } = useExtendedField(props);

   const isNumberField = props.controlProps ? props.controlProps.type === 'number' : false;

   const fixedOnChange = useCallback(
      (v: string) => {
         if (isNumberField) {
            const parsed = parseFloat(v);
            onChange(Number.isNaN(parsed) ? null : parsed);
         } else {
            onChange(v);
         }
      },
      [isNumberField, onChange],
   );

   const { setInputRef } = usePreventScrollForInputs();
   const value =
      typeof field.value === 'number' ? field.value.toString(10) : typeof field.value === 'string' ? field.value : '';

   const qa = props.qa ?? props.name.toString();

   return (
      <FieldLayout {...props}>
         {readonly ? (
            <div className={classes.readonlyValue}>
               {field.value || EMPTY_VALUE}
               <input type={'hidden'} name={field.name} value={value} />
            </div>
         ) : (
            <TextInput
               className={classNames(classes.control, { [styleHelpers.commonError]: showError })}
               controlRef={setInputRef}
               disabled={disabled}
               hasClear={!isNumberField}
               id={field.name}
               name={field.name}
               onBlur={onBlur}
               onUpdate={fixedOnChange}
               placeholder={props.placeholder}
               qa={qa}
               value={value}
               {...props.controlProps}
            />
         )}
      </FieldLayout>
   );
});

InputField.displayName = 'InputField';
