import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { object, string } from 'yup';

import { json } from '../../formatters';
import { ExtendedFieldConfig } from '../models';

import { SetField } from './SetField';

const initialValues = {
   empty: new Set(),
   regular: new Set(['value1', 'value2']),
   regularNumber: new Set([1, 2, 3]),
   withError: null,
};

type FormValue = typeof initialValues;

function SetFieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: string().required(),
         xField: string().min(10),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <SetField placeholder={'Placeholder'} {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

export const Regular = () => <SetFieldStory name={'regular'} label={'Regular'} />;

export const RegularMultiline = () => (
   <SetFieldStory name={'regular'} label={'Regular'} controlProps={{ separator: /\n/ }} />
);

export const RegularNumber = () => (
   <SetFieldStory name={'regularNumber'} label={'Regular'} controlProps={{ type: 'number' }} />
);

export const Disabled = () => <SetFieldStory name={'regular'} label={'Disabled'} disabled={true} />;

export const Readonly = () => <SetFieldStory name={'regular'} label={'Readonly'} readonly={true} />;

export const Empty = () => <SetFieldStory name={'empty'} label={'Readonly & Empty'} readonly={true} />;

export const WithError = () => <SetFieldStory name={'withError'} label={'With error'} />;

export const Playground = () => (
   <SetFieldStory
      name={'regular'}
      label={text('label', 'label')}
      placeholder={text('placeholder', 'placeholder')}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
   />
);

export default {
   title: 'forms2/fields/SetField',
} as Meta;
