import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { bool, object, string } from 'yup';

import { json } from '../../formatters';
import { ExtendedFieldConfig } from '../models';

import { SwitcherField } from './SwitcherField';

const initialValues = {
   empty: null as boolean | null,
   regular: true,
   withError: null as boolean | null,
};

type FormValue = typeof initialValues;

function SwitcherFieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         regular: bool().required(),
         withError: string().required(),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <SwitcherField {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

export const Regular = () => <SwitcherFieldStory name={'regular'}>Regular</SwitcherFieldStory>;

export const Disabled = () => (
   <SwitcherFieldStory name={'regular'} disabled={true}>
      Disabled
   </SwitcherFieldStory>
);

export const Readonly = () => (
   <SwitcherFieldStory name={'regular'} readonly={true}>
      Readonly
   </SwitcherFieldStory>
);

export const Indeterminate = () => <SwitcherFieldStory name={'empty'}>Indeterminate</SwitcherFieldStory>;

export const IndeterminateReadonly = () => (
   <SwitcherFieldStory name={'empty'} readonly={true}>
      IndeterminateReadonly
   </SwitcherFieldStory>
);

export const WithError = () => (
   <SwitcherFieldStory name={'withError'} label={'With error'}>
      WithError
   </SwitcherFieldStory>
);

export const Playground = () => (
   <SwitcherFieldStory
      name={'regular'}
      label={text('label', 'label')}
      placeholder={text('placeholder', 'placeholder')}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
   >
      {text('children', 'Switcher label')}
   </SwitcherFieldStory>
);

export default {
   title: 'forms2/fields/SwitcherField',
} as Meta;
