import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { Button } from '@yandex-cloud/uikit';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { object, string } from 'yup';

import { json } from '../../formatters';
import { ExtendedFieldConfig } from '../models';

import { TextControlProps, TextField } from './TextField';

const initialValues = {
   empty: '',
   regular: 'initialValue',
   withError: '',
};

type FormValue = typeof initialValues;

function TextFieldStory(props: ExtendedFieldConfig<FormValue, string, TextControlProps>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: string().required(),
         xField: string().min(10),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <TextField placeholder={'Placeholder'} {...props} />

               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

export const Regular = () => <TextFieldStory name={'regular'} label={'Regular'} />;

export const Disabled = () => <TextFieldStory name={'regular'} label={'Disabled'} disabled={true} />;

export const Readonly = () => <TextFieldStory name={'regular'} label={'Readonly'} readonly={true} />;

export const Empty = () => <TextFieldStory name={'empty'} label={'Readonly & Empty'} readonly={true} />;

export const WithError = () => <TextFieldStory name={'withError'} label={'With error'} />;

export const Playground = () => (
   <TextFieldStory
      name={'regular'}
      label={text('label', 'label')}
      placeholder={text('placeholder', 'placeholder')}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
      controlProps={{ monospaceFont: true, readonlyPre: boolean('readonlyPre', false) }}
   />
);

export default {
   title: 'forms2/fields/TextField',
} as Meta;
