import { FieldInputProps, FieldMetaProps, FormikProps, useField, useFormikContext } from 'formik';
import { useCallback, useContext, useMemo } from 'react';

import { CalculatedProp, ExtendedFieldConfig, FormContext, FormHooksContext } from './models';
import { OverridingFieldPropsContext, useNamePrefixer } from './wrappers';

export function getCalculatedValue<T>(
   prop: CalculatedProp<T, any> | undefined,
   form: FormikProps<any>,
   field: FieldInputProps<any>,
   context?: any,
): T | undefined {
   if (typeof prop === 'function') {
      // @ts-ignore // FIXME
      return prop(field.name, form, context);
   }
   return prop;
}

export function useCalculatedProp<T>(
   fieldName: ExtendedFieldConfig['name'],
   calculatedProp: CalculatedProp<T, any>,
): T | undefined {
   const formContext = useContext(FormContext);
   const form = useFormikContext();
   const [field] = useField(fieldName as string);

   return getCalculatedValue(calculatedProp, form, field, formContext);
}

interface UseExtendedFieldResult<V> {
   disabled: boolean;
   field: FieldInputProps<V>;
   meta: FieldMetaProps<V>;
   readonly: boolean;
   showError: boolean;

   onChange(v: V): void;

   onBlur(): void;
}

export function useExtendedField<V>(rawProps: ExtendedFieldConfig<any, V, any>): UseExtendedFieldResult<V> {
   const overridingProps = useContext(OverridingFieldPropsContext);
   const namePrefixer = useNamePrefixer();
   const props = useMemo(
      () => ({
         ...rawProps,
         ...overridingProps,
         name: namePrefixer(overridingProps?.name ?? rawProps.name),
      }),
      [overridingProps, namePrefixer, rawProps],
   );

   const formHooks = useContext(FormHooksContext);
   const context = useContext(FormContext);
   const form = useFormikContext();
   const [field, meta] = useField<V>(props as any); // TODO remove any

   const { name, onBlur, onChange } = field;

   const onBlurSimplified = useCallback(() => {
      onBlur({ target: { name } });

      if (formHooks !== null && formHooks.onFieldBlur) {
         formHooks.onFieldBlur(name);
      }
   }, [onBlur, name, formHooks]);

   const onChangeSimplified = useCallback(
      (value: V) => {
         onChange({ target: { name, value } });

         if (formHooks !== null && formHooks.onFieldChange) {
            formHooks.onFieldChange(name, value);
         }
      },
      [onChange, name, formHooks],
   );

   return {
      disabled: getCalculatedValue(props.disabled, form, field, context) || form.isSubmitting,
      field,
      meta,
      onBlur: onBlurSimplified,
      onChange: onChangeSimplified,
      readonly: Boolean(getCalculatedValue(props.readonly, form, field, context)),
      showError: Boolean(meta.touched && meta.error),
   };
}

/**
 * Строит функцию для удобного взятия подписей к полям для вывода их в валидационных ошибках
 *
 * @example
 * const fields: Array<ExtendedFieldConfig<Entity>> = [
 *    {name:'name', label:'Entity name'},
 *    {name:'email', label:'Email'}
 * ];
 *
 * const getLabel = getFieldLabels(fields);
 *
 * const validationSchema = yup.object<Entity>({
 *    name: string().label(getLabel('name')).required(),
 *    email: string().label(getLabel('email')).required(),
 * });
 */
export function getFieldLabels<T>(fields: ExtendedFieldConfig<T>[]): (name: keyof T) => string | null {
   const map = fields.reduce((acc, field) => {
      if (typeof field.label === 'string') {
         acc.set(field.name as string, field.label);
      }

      return acc;
   }, new Map<string, string>());

   return name => (map.has(name as string) ? map.get(name as string)! : null);
}

export function formatName(...parts: ExtendedFieldConfig['name'][]): string {
   return parts
      .map(p => (typeof p === 'number' ? `[${p}]` : p))
      .filter(Boolean)
      .join('.');
}
