import React, { ReactNode, useCallback, useMemo } from 'react';

import { AddButton, DeclarativeFields, FieldLayout, FormCard, FormCardProps } from '../../components';
import { formatName, useExtendedField } from '../../helpers';
import { ContainerControlProps, ExtendedFieldConfig } from '../../models';
import { ContainerWrapper } from '../../wrappers';

import classes from './CardsListContainer.module.css';

export interface CardsListContainerControlProps extends ContainerControlProps {
   addButtonText?: string;
   cardProps?: FormCardProps;
   keepAtLeastOne?: boolean;

   getCardTitle(record: any): ReactNode;
}

export const CardsListContainer: React.FC<ExtendedFieldConfig<any, any[], CardsListContainerControlProps>> = React.memo(
   props => {
      const {
         addButtonText = 'Add',
         cardProps,
         fields,
         getCardTitle,
         keepAtLeastOne = false,
      } = props.controlProps ?? {};
      const { field, readonly, disabled, onChange } = useExtendedField<any[]>(props);

      const data = useMemo(() => field?.value ?? [], [field?.value]);

      // region Handlers
      const handleAdd = useCallback(() => onChange([...data, {}]), [data, onChange]);

      const handleRemove = useCallback((record: any) => onChange(data.filter(r => r !== record)), [data, onChange]);
      // endregion

      return (
         <FieldLayout {...props}>
            {Array.from(data).map((record, i) => {
               const cardTitle: ReactNode = getCardTitle?.(record) || `#${i}`;
               const fieldNamePrefix = formatName(props.name, i);
               const canRemove = !disabled && !readonly && (keepAtLeastOne ? data.length > 1 : true);

               return (
                  <ContainerWrapper {...props} key={fieldNamePrefix} name={fieldNamePrefix}>
                     <FormCard
                        {...cardProps}
                        hasRemoveButton={canRemove}
                        onRemove={() => handleRemove(record)}
                        title={cardTitle}
                     >
                        {props.children ?? (
                           <DeclarativeFields configs={fields} defaults={{ readonly, disabled, readonlyDots: true }} />
                        )}
                     </FormCard>
                  </ContainerWrapper>
               );
            })}

            {!readonly && (
               <AddButton className={classes.addButton} disabled={disabled} onClick={handleAdd}>
                  {addButtonText}
               </AddButton>
            )}
         </FieldLayout>
      );
   },
);

CardsListContainer.displayName = 'CardsListContainer';
