import * as React from 'react';
import { useCallback, useMemo } from 'react';

import { isEmpty } from '../../../helpers';
import { RemoveButton } from '../../../small_components';
import { AddButton, DeclarativeFields, FieldLayout } from '../../components';
import { formatName, useExtendedField } from '../../helpers';
import { ContainerControlProps, ExtendedFieldConfig } from '../../models';
import { ContainerWrapper } from '../../wrappers';

import classes from './ListContainer.module.css';

export interface ListContainerControlProps extends ContainerControlProps {
   addButtonText?: string;
   keepAtLeastOne?: boolean;
}

export type ListContainerProps = ExtendedFieldConfig<any, any[], ListContainerControlProps>;

export const ListContainer: React.FC<ListContainerProps> = React.memo(props => {
   const { fields, addButtonText = 'Add', keepAtLeastOne = false } = props.controlProps ?? {};
   const { field, readonly, disabled, onChange } = useExtendedField<any[]>(props);

   const data = useMemo(() => field?.value ?? [], [field?.value]);

   // region Handlers
   const handleAdd = useCallback(() => onChange([...data, null]), [data, onChange]);

   const handleRemove = useCallback((record: any) => onChange(data.filter(r => r !== record)), [data, onChange]);
   // endregion

   return (
      <FieldLayout {...props}>
         {!isEmpty(data) && (
            <div className={classes.wrapper}>
               {Array.from(data).map((record, i) => {
                  const fieldNamePrefix = formatName(props.name, i);
                  const canRemove = !readonly && (keepAtLeastOne ? data.length > 1 : true);

                  return (
                     <ContainerWrapper {...props} key={fieldNamePrefix} name={fieldNamePrefix} className={classes.row}>
                        {props.children ?? <DeclarativeFields configs={fields} fieldClassName={classes.itemField} />}

                        {canRemove && (
                           <RemoveButton
                              className={classes.removeButton}
                              disabled={disabled}
                              onClick={() => handleRemove(record)}
                           />
                        )}
                     </ContainerWrapper>
                  );
               })}
            </div>
         )}

         {!readonly && (
            <AddButton className={classes.addButton} disabled={disabled} onClick={handleAdd}>
               {addButtonText}
            </AddButton>
         )}
      </FieldLayout>
   );
});

ListContainer.displayName = 'ListContainer';
