import { Table, TableColumnConfig, TableProps } from '@yandex-cloud/uikit';
import { Field } from 'formik';
import * as React from 'react';
import { ReactElement, useCallback, useMemo } from 'react';

import { RemoveButton } from '../../../small_components';
import { AddButton, FieldLayout } from '../../components';
import { formatName, useExtendedField } from '../../helpers';
import { ContainerControlProps, ExtendedFieldConfig } from '../../models';
import { ContainerWrapper, FieldOverrideWrapper, NamePrefixingWrapper } from '../../wrappers';

import classes from './TableContainer.module.css';

export interface TableContainerControlProps extends ContainerControlProps {
   addButtonText?: string;
   keepAtLeastOne?: boolean;
   tableProps?: Omit<TableProps<any>, 'columns' | 'data' | 'className'>;
}

export type TableContainerProps = ExtendedFieldConfig<any, any[], TableContainerControlProps>;

export const TableContainer: React.FC<TableContainerProps> = React.memo(props => {
   const { fields, addButtonText = 'Add row', keepAtLeastOne = false } = props.controlProps ?? {};
   const { field, readonly, disabled, onChange } = useExtendedField<any[]>(props);

   const data = useMemo(() => field?.value ?? [], [field?.value]);

   // region Handlers
   const handleAdd = useCallback(() => onChange([...data, {}]), [data, onChange]);

   const handleRemove = useCallback((record: any) => onChange(data.filter(r => r !== record)), [data, onChange]);
   // endregion

   const dataColumns = useMemo(() => {
      if (fields) {
         return fields.map(
            fieldProps =>
               ({
                  ...fieldProps.controlProps?.columnProps,
                  id: fieldProps.name,
                  name: () => fieldProps.label,
                  template: (item, i) => <Field {...fieldProps} name={formatName(i, fieldProps.name)} label={null} />,
               } as TableColumnConfig<any>),
         );
      }

      return (
         React.Children.map(props.children, child => {
            const fieldProps = (child as ReactElement).props as ExtendedFieldConfig;

            return {
               ...fieldProps.controlProps?.columnProps,
               id: fieldProps.name,
               name: () => fieldProps.label,
               template: (item, i) => (
                  <NamePrefixingWrapper containerName={formatName(i)}>
                     <FieldOverrideWrapper label={null}>{child}</FieldOverrideWrapper>
                  </NamePrefixingWrapper>
               ),
            } as TableColumnConfig<any>;
         }) ?? []
      );
   }, [fields, props.children]);

   const columns = useMemo(() => {
      const canRemove = !readonly && (keepAtLeastOne ? data.length > 1 : true);

      const actionColumn: TableColumnConfig<any> = {
         id: 'actions',
         name: '',
         template: item => canRemove && <RemoveButton onClick={() => handleRemove(item)} disabled={disabled} />,
         width: 20,
      };

      return [...dataColumns, actionColumn];
   }, [data.length, disabled, dataColumns, handleRemove, keepAtLeastOne, readonly]);

   return (
      <FieldLayout {...props}>
         <ContainerWrapper {...props}>
            <Table columns={columns} data={data} className={classes.table} {...props.controlProps?.tableProps} />
         </ContainerWrapper>

         {!readonly && (
            <AddButton disabled={disabled} onClick={handleAdd}>
               {addButtonText}
            </AddButton>
         )}
      </FieldLayout>
   );
});

TableContainer.displayName = 'TableContainer';
