import { FieldAttributes, FormikProps } from 'formik';
import React, { ReactNode } from 'react';

export interface FormHooks<T> {
   onFieldChange?<K extends keyof T>(field: K, value: T[K]): void;

   onFieldBlur?<K extends keyof T>(field: K): void;
}

/**
 * Обходной способ достучатся до события изменения значения в поле извне
 *
 * Вынужден сделать через Context API, т.к. Formik не предоставляет способов сделать нормально
 *
 * Работает в комбинации с хуком useExtendedField
 */
export const FormHooksContext = React.createContext<FormHooks<any> | null>(null);

/**
 * Контекст формы. Доступен при вычислении свойств:
 * required, disabled, readonly, help, hidden
 */
export const FormContext = React.createContext<any>(null);

export type CalculatedProp<T, FormValue> =
   | T
   | ((fieldName: string, f: FormikProps<FormValue>, context?: any) => T | null);

export interface CalculatedFieldProps<FormValue> {
   disabled?: CalculatedProp<boolean, FormValue>;
   help?: CalculatedProp<ReactNode, FormValue>;
   hidden?: CalculatedProp<boolean, FormValue>;
   readonly?: CalculatedProp<boolean, FormValue>;
   required?: CalculatedProp<boolean, FormValue>;
}

export interface RoundedInputControlProps {
   baseMeasure?: string;
   useRenderDetails?: boolean;
}

// eslint-disable-next-line @typescript-eslint/no-unused-vars
export interface ExtendedFieldConfig<FormValue = any, V = any, ControlProps = Record<string, any>>
   extends CalculatedFieldProps<FormValue>,
      FieldAttributes<any> {
   /**
    * Большой лейбл
    */
   bigLabel?: boolean;

   controlProps?: ControlProps;

   /**
    * Скрывает вывод валидационных ошибок.
    *
    * Подразумевается, что ошибки будут показаны как-то кастомно.
    */
   hideErrors?: boolean;

   /**
    * Подсказка к полю, пояснение к подписи
    */
   hint?: ReactNode;

   /**
    * Key for using as "key" when React renders list of fields inside <DeclarativeFields>
    *
    * If skipped, the "name" will be used for "key".
    *
    * @see https://reactjs.org/docs/lists-and-keys.html#keys
    */
   key?: string;

   /**
    * Подпись к полю. Обычно слева.
    */
   label?: ReactNode;

   /**
    * Класс для лейбла
    */
   labelClassName?: string;

   /**
    * Имя поля, оно же – атрибут `id`
    *
    * Может быть как просто property, так сложным путем, поддерживающегося в _.set, _.get
    */
   name: Exclude<keyof FormValue, symbol> | string;

   qa?: string;

   /**
    * Точки в readonly режиме
    */
   readonlyDots?: boolean;
}

export type LabelPosition = 'left' | 'top';

export interface ContainerConfig<FormValue, V> {
   /**
    * Props of internal <FieldLayout>
    */
   fieldLayoutProps?: Omit<ExtendedFieldConfig<FormValue, V, void>, 'as' | 'controlProps' | 'key' | 'label' | 'name'>;

   /**
    * Container internal fields
    */
   fields: ExtendedFieldConfig<FormValue, V>[];

   /**
    * Key for using as "key" when React renders list.
    *
    * @see https://reactjs.org/docs/lists-and-keys.html#keys
    */
   key: string;

   /**
    * Main label (usual it is just a <FieldLayout> label, but sometimes differs)
    */
   label?: ReactNode;

   /**
    * Label position for internal fields.
    *
    * @default left
    */
   labelPosition?: LabelPosition;

   /**
    * "name" of the internal <FieldLayout> wrapper.
    *
    * Use to highlight validation errors.
    */
   name: ExtendedFieldConfig<FormValue>['name'];

   /**
    * Prevent adding container's name to internal fields
    */
   preventNamePrefixing?: boolean;
}

export interface ContainerControlProps {
   /**
    * Used only if rendered by <DeclarativeFields>
    */
   fields?: ExtendedFieldConfig[];

   /**
    * Label position for internal fields.
    *
    * @default left
    */
   labelPosition?: LabelPosition;

   /**
    * Prevent adding container's name to internal fields
    */
   preventNamePrefixing?: boolean;
}
