import { Meta } from '@storybook/react';
import { Button, TabsDirection } from '@yandex-cloud/uikit';
import { Form, Formik, FormikProps } from 'formik';
import React, { useCallback, useState } from 'react';

import { Loader } from '../../small_components';
import { SuggestLayer } from '../../suggestions';
import { DeclarativeFields, DevForm, DevFormValues } from '../components';
import { advanced, card, fieldset, row, tabs } from '../containers';
import { DateTimeField, EnumField, InputField, SuggestSelectField, SwitcherField, TextField } from '../fields';
import { cards, list, table } from '../listContainers';
import { ExtendedFieldConfig } from '../models';

import { DemoFormValues, demoInitialValues, demoSexOptions, demoValidationSchema, FamilyMember } from './demo.form';

const nameField: ExtendedFieldConfig<DemoFormValues> = {
   as: InputField,
   label: 'Name',
   name: 'name',
   placeholder: 'Name',
   required: true,
};

const birthdayField: ExtendedFieldConfig<DemoFormValues> = {
   as: DateTimeField,
   name: 'birthday',
   label: 'Birthday',
   required: true,
};

const fields: ExtendedFieldConfig<DemoFormValues>[] = [
   tabs({
      direction: TabsDirection.Vertical,
      fieldLayoutProps: {
         bigLabel: true,
         hint: '[TabsContainer]',
      },
      key: '_tabs',
      label: 'Declarative form Demo',
      labelPosition: 'top',
      name: '_tabs',
      preventNamePrefixing: true,
      tabConfigs: [
         {
            id: '_personal',
            title: 'Personal',
            fields: [
               card({
                  cardProps: {
                     subTitle: '[CardContainer]',
                  },
                  fields: [
                     nameField,
                     birthdayField,
                     {
                        as: EnumField,
                        controlProps: {
                           options: demoSexOptions,
                        },
                        label: 'Sex',
                        name: 'sex',
                        required: true,
                     },
                     {
                        as: InputField,
                        controlProps: {
                           type: 'email',
                        },
                        label: 'Email',
                        name: 'email',
                        placeholder: 'Email',
                        required: true,
                        hint: 'Personal or work email - any so that you can be contacted',
                     },
                     {
                        as: TextField,
                        label: 'Biography',
                        name: 'bio',
                        placeholder: 'Enter short biography',
                        required: true,
                     },
                  ],
                  key: '_personalCard',
                  name: '_personalCard',
                  label: 'Personal info',
                  labelPosition: 'top',
                  preventNamePrefixing: true,
               }),
            ],
         },
         {
            id: '_family',
            title: 'Family',
            fields: [
               fieldset({
                  fieldLayoutProps: {
                     hint: '[FieldsetContainer]',
                  },
                  fields: [
                     row({
                        fieldLayoutProps: {
                           help: '[RowContainer] above',
                        },
                        fields: [nameField, birthdayField],
                        key: 'spouse',
                        label: null,
                        name: '_spouseRow',
                        preventNamePrefixing: true,
                     }),
                  ],
                  key: 'spouse',
                  label: 'Spouse',
                  name: 'spouse',
               }),
               fieldset({
                  fieldLayoutProps: {
                     hint: '[FieldsetContainer]',
                  },
                  fields: [
                     {
                        as: SwitcherField,
                        label: 'Has children',
                        name: 'hasChildren',
                        required: true,
                     },
                     cards({
                        addButtonText: 'Add child',
                        fieldLayoutProps: {
                           bigLabel: true,
                           hint: '[CardsListContainer]',
                           hidden: (name, formik) => !formik.values.family?.hasChildren,
                        } as Partial<ExtendedFieldConfig<DemoFormValues>>,
                        fields: [nameField, birthdayField],
                        getCardTitle: (p: FamilyMember | undefined) => p?.name,
                        key: 'children',
                        label: 'Children',
                        name: 'children',
                     }),
                  ],
                  key: 'children',
                  label: 'Children',
                  labelPosition: 'left',
                  name: '_childrenFieldset',
                  preventNamePrefixing: true,
               }),
            ],
            name: 'family',
         },
         {
            id: '_work',
            title: 'Work',
            fields: [
               table({
                  addButtonText: 'Add workplace',
                  fieldLayoutProps: {
                     bigLabel: true,
                     hint: '[TableContainer]',
                  },
                  fields: [
                     { as: InputField, name: 'companyName', label: 'Company name' },
                     { as: DateTimeField, name: 'startDate', label: 'Start date' },
                     { as: DateTimeField, name: 'endDate', label: 'End date' },
                  ],
                  key: 'workplaces',
                  label: 'Workplaces',
                  name: 'workplaces',
               }),
               list({
                  addButtonText: 'Add skill',
                  fieldLayoutProps: { required: true, bigLabel: true, hint: '[ListContainer]' },
                  fields: [{ as: InputField, label: null, name: '' }],
                  key: 'skills',
                  label: 'Skills',
                  name: 'skills',
               }),
               advanced({
                  fields: [
                     {
                        as: SuggestSelectField,
                        label: 'ABC service id',
                        name: 'abcServiceId',
                        controlProps: {
                           layer: SuggestLayer.Services,
                        },
                        required: true,
                     },
                  ],
                  formAdvancedSectionProps: { toggleText: 'Open', title: 'This is title of <FormAdvancedSection>' },
                  key: '_advanced',
                  label: 'Additional info',
                  name: '_advanced',
                  preventNamePrefixing: true,
               }),
            ],
         },
      ],
   }),
];

function DeclarativeFormExample(form: FormikProps<DemoFormValues>) {
   const style = {
      // '--forms-field-layout': 'column',
   } as any;

   return (
      <Form style={style}>
         <DeclarativeFields<DemoFormValues> configs={fields} />

         <br />
         <Button type={'reset'} disabled={form.isSubmitting}>
            reset
         </Button>

         <Button view={'action'} type={'submit'} disabled={form.isSubmitting}>
            submit
         </Button>

         <Loader visible={form.isSubmitting} inline={true} text={'Form is submitting'} />

         <DevFormValues />

         <DevForm />
      </Form>
   );
}

export const DeclarativeDemo = () => {
   const [value, setValue] = useState<DemoFormValues>(demoInitialValues);

   const onSubmit = useCallback(
      (v: DemoFormValues) =>
         new Promise(resolve => {
            setTimeout(() => {
               setValue(v);
               resolve(undefined);
            }, 5000);
         }),
      [],
   );

   return (
      <Formik
         initialValues={value}
         onSubmit={onSubmit}
         component={DeclarativeFormExample}
         validationSchema={demoValidationSchema}
      />
   );
};

export default {
   title: 'forms2/Declarative Demo',
} as Meta;
