import { Meta } from '@storybook/react';
import { Button, TabsDirection } from '@yandex-cloud/uikit';
import { Form, Formik, FormikProps } from 'formik';
import React, { useCallback, useRef, useState } from 'react';

import { Loader } from '../../small_components';
import { SuggestLayer } from '../../suggestions';
import { DevForm, DevFormValues } from '../components';
import { AdvancedContainer, CardContainer, FieldsetContainer, RowContainer, TabsContainer } from '../containers';
import { DateTimeField, EnumField, InputField, SuggestSelectField, SwitcherField, TextField } from '../fields';
import { CardsListContainer, ListContainer, TableContainer } from '../listContainers';

import { DemoFormValues, demoInitialValues, demoSexOptions, demoValidationSchema, FamilyMember } from './demo.form';

export const ReactWayDemo = () => {
   const formRef = useRef<FormikProps<DemoFormValues>>();
   const [value, setValue] = useState(demoInitialValues);

   const onSubmit = useCallback(
      (v: DemoFormValues) =>
         new Promise(resolve => {
            setTimeout(() => {
               setValue(v);
               if (formRef.current) {
                  formRef.current.resetForm({ values: v });
               }
               resolve(undefined);
            }, 5000);
         }),
      [],
   );

   const nameField = <InputField label={'Name'} name={'name'} placeholder={'Name'} required={true} />;

   const birthdayField = <DateTimeField name={'birthday'} label={'Birthday'} required={true} />;

   return (
      <Formik initialValues={value} onSubmit={onSubmit} validationSchema={demoValidationSchema}>
         {form => {
            formRef.current = form;

            return (
               <Form>
                  <TabsContainer
                     name={'_tabs'}
                     controlProps={{
                        direction: TabsDirection.Vertical,
                        labelPosition: 'top',
                        preventNamePrefixing: true,
                     }}
                     bigLabel={true}
                     hint={'[TabsContainer]'}
                  >
                     <TabsContainer.Tab id={'_personal'} title={'Personal'}>
                        <CardContainer
                           name={'_personalCard'}
                           label={'Personal info'}
                           controlProps={{
                              preventNamePrefixing: true,
                              cardProps: {
                                 subTitle: '[CardContainer]',
                              },
                           }}
                        >
                           {nameField}
                           {birthdayField}
                           <EnumField
                              label={'Sex'}
                              name={'sex'}
                              required={true}
                              controlProps={{
                                 options: demoSexOptions,
                              }}
                           />
                           <InputField
                              label={'Email'}
                              name={'email'}
                              placeholder={'Email'}
                              controlProps={{ type: 'email' }}
                              required={true}
                              hint={'Personal or work email - any so that you can be contacted'}
                           />
                           <TextField
                              label={'Biography'}
                              name={'bio'}
                              placeholder={'Enter short biography'}
                              required={true}
                           />
                        </CardContainer>
                     </TabsContainer.Tab>

                     <TabsContainer.Tab id={'_family'} title={'Family'} name={'family'}>
                        <FieldsetContainer name={'spouse'} label={'Spouse'} hint={'[FieldsetContainer]'}>
                           <RowContainer name={'_spouseRow'} label={null} controlProps={{ preventNamePrefixing: true }}>
                              {nameField}
                              {birthdayField}
                           </RowContainer>
                        </FieldsetContainer>

                        <FieldsetContainer
                           name={'_childrenFieldset'}
                           label={'Children'}
                           controlProps={{
                              labelPosition: 'left',
                              preventNamePrefixing: true,
                           }}
                        >
                           <SwitcherField name={'hasChildren'} label={'Has children'} required={true} />

                           <CardsListContainer
                              name={'children'}
                              label={'Children'}
                              bigLabel={true}
                              hint={'[CardsListContainer]'}
                              hidden={(name, formik: FormikProps<DemoFormValues>) => !formik.values.family?.hasChildren}
                              controlProps={{
                                 getCardTitle: (p: FamilyMember | undefined) => p?.name,
                              }}
                           >
                              {nameField}
                              {birthdayField}
                           </CardsListContainer>
                        </FieldsetContainer>
                     </TabsContainer.Tab>

                     <TabsContainer.Tab id={'_work'} title={'Work'}>
                        <TableContainer
                           name={'workplaces'}
                           label={'Workplaces'}
                           bigLabel={true}
                           hint={'[TableContainer]'}
                        >
                           <InputField name={'companyName'} label={'Company name'} />
                           <DateTimeField name={'startDate'} label={'Start date'} />
                           <DateTimeField name={'endDate'} label={'End date'} />
                        </TableContainer>

                        <ListContainer
                           name={'skills'}
                           label={'Skills'}
                           bigLabel={true}
                           hint={'[ListContainer]'}
                           required={true}
                        >
                           <InputField name={''} label={null} />
                        </ListContainer>

                        <AdvancedContainer
                           name={'_advanced'}
                           label={'Additional info'}
                           controlProps={{
                              preventNamePrefixing: true,
                              formAdvancedSectionProps: {
                                 toggleText: 'Open',
                                 title: 'This is title of <FormAdvancedSection>',
                              },
                           }}
                        >
                           <SuggestSelectField
                              label={'ABC service id'}
                              name={'abcServiceId'}
                              required={true}
                              controlProps={{
                                 layer: SuggestLayer.Services,
                              }}
                           />
                        </AdvancedContainer>
                     </TabsContainer.Tab>
                  </TabsContainer>

                  <Button type={'reset'} disabled={form.isSubmitting}>
                     reset
                  </Button>

                  <Button view={'action'} type={'submit'} disabled={form.isSubmitting}>
                     submit
                  </Button>

                  <Loader visible={form.isSubmitting} inline={true} text={'Form is submitting'} />

                  <DevFormValues />

                  <DevForm />
               </Form>
            );
         }}
      </Formik>
   );
};
ReactWayDemo.storyName = 'React-way Demo';

export default {
   title: 'forms2/React-way Demo',
} as Meta;
