import { array, bool, date, number, object, ObjectSchemaDefinition, string } from 'yup';

interface Workplace {
   companyName: string;
   endDate?: Date;
   startDate: Date;
}

export interface FamilyMember {
   birthday: Date;
   name: string;
   sex: Sex;
}

type Sex = 'male' | 'female';

export interface DemoFormValues extends FamilyMember {
   abcServiceId: number | null;
   bio: string;
   email: string;
   family: {
      spouse: FamilyMember;
      hasChildren: boolean;
      children: FamilyMember[];
   };
   skills: (string | null)[];
   workplaces: Workplace[];
}

export const demoInitialValues: DemoFormValues = {
   // TAB Personal
   // input
   name: 'Viktor',

   // date
   birthday: new Date('1987-07-07'),

   // enum
   sex: 'male',

   // input
   email: 'viktor@viktor.com',

   // textarea
   bio: 'Short biography',

   // TAB Family
   // fieldset
   family: {
      // row
      spouse: {
         name: 'Elena',
         birthday: new Date('1985-03-06'),
         sex: 'female',
      },
      // cards
      children: [
         { name: 'Ivan', sex: 'male', birthday: new Date('2005-01-13') },
         { name: 'Olga', sex: 'female', birthday: new Date('2020-06-18') },
      ],
      hasChildren: true,
   },

   // TAB Work
   // table
   workplaces: [
      { companyName: 'DV-bank', startDate: new Date('2008-09-01'), endDate: new Date('2010-05-25') },
      { companyName: 'Encraft', startDate: new Date('2010-06-17'), endDate: new Date('2014-06-10') },
      { companyName: 'Emply', startDate: new Date('2014-08-10'), endDate: new Date('2015-10-01') },
      { companyName: 'Encraft', startDate: new Date('2015-10-01'), endDate: new Date('2018-07-20') },
      { companyName: 'Yandex', startDate: new Date('2018-09-19') },
   ],
   // list
   skills: ['javascript', 'typescript'],
   // advanced
   abcServiceId: 3494 as number | null,
};

const familyMemberSchemeDef: ObjectSchemaDefinition<FamilyMember> = {
   name: string(),
   birthday: date().typeError('Must be a Date'),
   sex: string().oneOf(['male', 'female'], 'Please choice properly sex').required() as any,
};

const familyMemberScheme = object(familyMemberSchemeDef);

const familyScheme = object<DemoFormValues['family']>({
   children: array(familyMemberScheme),
   hasChildren: bool().required(),
   spouse: familyMemberScheme,
});

const workplaceScheme = object<Workplace>({
   endDate: date().notRequired(),
   startDate: date(),
   companyName: string().required(),
});

export const demoValidationSchema = object<DemoFormValues>({
   ...familyMemberSchemeDef,
   abcServiceId: number().nullable().required(),
   bio: string().required('Required'),
   family: familyScheme,
   email: string().email('Invalid email address').required(),
   birthday: date(),
   name: string(),
   skills: array(string().nullable()),
   workplaces: array(workplaceScheme),
});

export const demoSexOptions = [
   { value: 'male', title: 'Male' },
   { value: 'female', title: 'Female' },
   { value: 'unknown', title: 'Unknown' },
];
