import { IClonable } from '../../_models';

import { deepClone } from './deepClone';

describe('deepClone', () => {
   it('should clone string', () => {
      expect(deepClone('xxx')).toBe('xxx');
   });

   it('should clone bool', () => {
      expect(deepClone(true)).toBe(true);
   });

   it('should clone numbers', () => {
      expect(deepClone(1e6)).toBe(1000000);
      expect(deepClone(0.0000002)).toBe(0.0000002);
   });

   it('should clone null', () => {
      expect(deepClone(null)).toBe(null);
   });

   it('should clone undefined', () => {
      expect(deepClone(undefined)).toBe(undefined);
   });

   it('should clone plain array', () => {
      expect(deepClone(['s', 12, null])).toEqual(['s', 12, null]);
   });

   it('should clone dates', () => {
      expect(deepClone(new Date('2018-11-12T04:43:44Z'))).toEqual(new Date('2018-11-12T04:43:44Z'));
   });

   it('should clone regexps', () => {
      expect(deepClone(/xx/)).toEqual(/xx/);
      expect(deepClone(/xx/i)).toEqual(/xx/i);
   });

   it('should clone sets', () => {
      expect(deepClone(new Set([1, 2]))).toEqual(new Set([1, 2]));
   });

   it('should clone maps', () => {
      const map = new Map([
         [1, 2],
         [3, 4],
      ]);

      expect(deepClone(map)).not.toBe(map);

      expect(deepClone(map)).toEqual(
         new Map([
            [1, 2],
            [3, 4],
         ]),
      );
   });

   it('should clone plain objects', () => {
      const input = { x: 1, y: new Date('2018-11-12T04:43:44Z') };
      const output = { y: new Date('2018-11-12T04:43:44Z'), x: 1 };

      expect(deepClone(input)).not.toBe(input);
      expect(deepClone(input)).toEqual(output);
   });

   class Item implements IClonable<Item> {
      public x = 2;

      public y = 3;

      public clone() {
         const item = new Item();
         item.x = this.x;
         item.y = this.y;

         return item;
      }
   }

   it('should clone object with clone method', () => {
      const input = { obj: new Item() };
      input.obj.x = 111;

      const cloned = deepClone(input);

      expect(cloned.obj).not.toBe(input);
      expect(cloned.obj.x).toBe(111);
      expect(cloned.obj).toBeInstanceOf(Item);
   });
});
