import { IEqualable } from '../../_models';

import { isEqual } from './isEqual';

describe('isEqual', () => {
   it('should work with numbers', () => {
      expect(isEqual(1, 1)).toBe(true);

      expect(isEqual(1, 2)).toBe(false);
      expect(isEqual(1, '1' as any)).toBe(false);
   });

   it('should work with arrays', () => {
      expect(isEqual([1], [1])).toBe(true);
      expect(isEqual([1, { x: 2 }], [1, { x: 2 }])).toBe(true);

      expect(isEqual([1, { x: 2 }], [1, { x: 3 }])).toBe(false);
      expect(isEqual([1], [1, 2])).toBe(false);
      expect(isEqual([1, 3], [1, 2])).toBe(false);
   });

   it('should work with objects', () => {
      expect(isEqual({}, {})).toBe(true);
      expect(isEqual({ x: 1, y: 2 }, { y: 2, x: 1 })).toBe(true);

      expect(isEqual({ x: 1, z: 2 }, { y: 2, x: 1 })).toBe(false);
   });

   it('should work with sets', () => {
      expect(isEqual(new Set([]), new Set([]))).toBe(true);

      expect(isEqual(new Set([{ x: 3 }, ['hello']]), new Set([{ x: 3 }, ['hello']]))).toBe(true);

      expect(isEqual(new Set([1]), new Set([2]))).toBe(false);

      expect(isEqual(new Set([2, 2, 3]), new Set([2, 2, 4]))).toBe(false);
   });

   it('should work with sets, ignoring order', () => {
      expect(isEqual(new Set([2, 3, 4]), new Set([2, 4, 3]))).toBe(true);

      expect(isEqual(new Set(['a', 'b', 'c']), new Set(['c', 'a', 'b']))).toBe(true);
   });

   it('should work with regexps', () => {
      expect(isEqual(/xxx/, null)).toBe(false);
      expect(isEqual(/xxx/, undefined)).toBe(false);
      expect(isEqual(/xxx/, {} as any)).toBe(false);
      expect(isEqual(/xxx/, [] as any)).toBe(false);
      expect(isEqual(/xxx/, new Set() as any)).toBe(false);

      expect(isEqual(/xxx/, /xxx/)).toBe(true);
      // eslint-disable-next-line prefer-regex-literals
      expect(isEqual(new RegExp('xxx'), /xxx/)).toBe(true);
      // eslint-disable-next-line prefer-regex-literals
      expect(isEqual(new RegExp('xxx', 'g'), /xxx/g)).toBe(true);

      expect(isEqual(/xxx/, /yyy/)).toBe(false);
      expect(isEqual(/xxx/i, /xxx/)).toBe(false);
      expect(isEqual(/xxx/g, /xxx/)).toBe(false);
   });

   it('should work with dates', () => {
      expect(isEqual(new Date(2018), new Date(2018))).toBe(true);

      expect(isEqual(new Date(2018), new Date(2017))).toBe(false);
   });

   it('should work with equalable classes', () => {
      class A implements IEqualable<A> {
         // @ts-ignore
         // noinspection JSUnusedLocalSymbols
         constructor(private x: number, private y: number) {}

         public isEqual(obj: A): boolean {
            return obj.x === this.x;
         }
      }

      expect(isEqual(new A(2, 5), new A(2, 3))).toBe(true);
      expect(isEqual(new A(2, 5), new A(3, 3))).not.toBe(true);
   });
});
