import { isEqualable } from '../../_models';

const primitiveTypes = new Set(['string', 'boolean', 'number']);

/**
 * Глубокая проверка на равенство двух объектов.
 *
 * Реализация в лоб, без поиска подводных камней.
 */
export function isEqual<T>(obj1: T, obj2: T): boolean {
   if (obj1 === obj2) {
      return true;
   }

   if (typeof obj1 !== typeof obj2) {
      return false;
   }

   if (primitiveTypes.has(typeof obj1)) {
      return obj1 === obj2;
   }

   if (Boolean(obj1) !== Boolean(obj2)) {
      return false;
   }

   if (isEqualable(obj1)) {
      return obj1.isEqual(obj2);
   }

   // null - тоже объект!
   if ((obj1 === null && obj2 !== null) || (obj1 != null && obj2 === null)) {
      return false;
   }

   if ((obj1 && (obj1 as any).constructor) !== (obj2 && (obj2 as any).constructor)) {
      return false;
   }

   if (obj1 instanceof RegExp && obj2 instanceof RegExp) {
      return obj1.toString() === obj2.toString();
   }

   if (obj1 instanceof Date && obj2 instanceof Date) {
      return obj1.getTime() === obj2.getTime();
   }

   if (obj1 instanceof Array && obj2 instanceof Array) {
      if (obj1.length !== obj2.length) {
         return false;
      }

      for (let i = 0; i < obj1.length; i += 1) {
         if (!isEqual(obj1[i], obj2[i])) {
            return false;
         }
      }

      return true;
   }

   // noinspection SuspiciousTypeOfGuard
   if (obj1 instanceof Set && obj2 instanceof Set) {
      if (obj1.size !== obj2.size) {
         return false;
      }
      const a1 = Array.from(obj1);
      const a2 = Array.from(obj2);

      return a1.every(item => a2.some(i => isEqual(i, item)));
   }

   if (typeof obj1 === 'object') {
      for (const field in obj1) {
         if ((obj1 as any).hasOwnProperty(field)) {
            // Проверка на совпадения списка полей
            if (!(obj2 as any).hasOwnProperty(field)) {
               return false;
            }

            // Проверка на идентичность значений
            if (!isEqual(obj1[field], obj2[field])) {
               return false;
            }
         }
      }
      // Проверка на совпадения списка полей в обратную сторону
      for (const field in obj2) {
         if ((obj2 as any).hasOwnProperty(field)) {
            if (!(obj1 as any).hasOwnProperty(field)) {
               return false;
            }
         }
      }

      return true;
   }

   // console.warn('Cannot detect equality for', obj1, obj2);

   return false;
}
