import { leftPad } from '../string/pad';

interface IDateParser {
   pattern: RegExp;

   parser(v: string): number;
}

function toDateString(year: number, month: number, day: number): string {
   const date = [year.toString(), leftPad(month.toString(), 2, '0'), leftPad(day.toString(), 2, '0')].join('-');

   // Добавил время, чтобы дата воспринималась, как локальная
   return `${date} 0:00:00`;
}

const DATE_FORMATS: IDateParser[] = [
   // Timestamp
   {
      parser: value => parseInt(value, 10) * 1000,
      pattern: /^\d+$/,
   },
   // Технический вариант YYYY-MM-DD
   {
      parser: value => {
         return Date.parse(`${value} 0:00:00`);
      },
      pattern: /^\d\d\d\d-\d\d-\d\d$/,
   },

   // Русский вариант DD.MM.YYYY (D.M.YY, D.M.YYYY, D.MM.YY, D.MM.YYYY, DD.M.YY, DD.M.YYYY, DD.MM.YY)
   {
      parser: value => {
         const [day, month, yearRaw] = value.split('.').map(p => parseInt(p, 10));
         let year = yearRaw;
         if (yearRaw < 100) {
            const currentYear = new Date().getFullYear();
            year = currentYear - (currentYear % 1000) + yearRaw;
         }

         return Date.parse(toDateString(year, month, day));
      },
      pattern: /^(\d){1,2}\.(\d){1,2}\.((\d){2}|(\d){4})$/,
   },

   // Английский вариант MM/DD/YYYY (M/D/YY, M/D/YYYY, M/DD/YY, M/DD/YYYY, MM/D/YY, MM/D/YYYY/, MM/DD/YY)
   {
      parser: value => {
         const [month, day, yearRaw] = value.split('/').map(p => parseInt(p, 10));
         let year = yearRaw;
         if (yearRaw < 100) {
            const currentYear = new Date().getFullYear();
            year = currentYear - (currentYear % 1000) + yearRaw;
         }

         return Date.parse(toDateString(year, month, day));
      },
      pattern: /^(\d){1,2}\/(\d){1,2}\/((\d){2}|(\d){4})$/,
   },
];

export function parseDate(rawValue: string): Date {
   let ts: number = Number.NaN;

   for (const { pattern, parser } of DATE_FORMATS) {
      if (pattern.test(rawValue)) {
         ts = parser(rawValue);
         break;
      }
   }

   return new Date(ts);
}
