import { BehaviorSubject, Observable } from 'rxjs';

import { Keys } from '../_models';
import { autobind } from '../utils';

import { IHotKey, IKey } from './models';

class GlobalHotKeys {
   public configs: BehaviorSubject<IHotKey[]>;

   public helpObs: Observable<boolean>;

   private initialized = false;

   private keySymbols = new Map<string, string>([
      [Keys.ArrowLeft, '←'],
      [Keys.ArrowRight, '→'],
      [Keys.Slash, '/'],
   ]);

   private visibleHelp: BehaviorSubject<boolean>;

   constructor() {
      const openHelpHotKey: IHotKey = {
         action: this.toggleHelp,
         help: 'Show this help',
         hotKey: { code: Keys.Slash, shiftKey: true },
      };

      this.configs = new BehaviorSubject([openHelpHotKey]);

      this.visibleHelp = new BehaviorSubject<boolean>(false);
      this.helpObs = this.visibleHelp.asObservable();
   }

   public init() {
      if (this.initialized) {
         return;
      }
      this.initialized = true;

      window.document.addEventListener('keydown', (e: KeyboardEvent) => {
         // console.log('keydown', e); // Для отладки

         const tag = (e.target as HTMLElement).tagName.toLowerCase();
         if (tag === 'textarea' || tag === 'input') {
            return;
         }

         for (const hotKey of this.configs.value) {
            if (this.isMatched(e, hotKey.hotKey)) {
               e.preventDefault();

               console.log(`HotKey: ${hotKey.help}`);
               hotKey.action(e);
            }
         }
      });
   }

   public getActiveHotKeys(): IHotKey[] {
      return this.configs.value;
   }

   @autobind
   public toggleHelp(): void {
      this.visibleHelp.next(!this.visibleHelp.value);
   }

   public register(...keys: IHotKey[]): () => void {
      const hotKeys = [...this.configs.value, ...keys];
      this.configs.next(hotKeys);

      return () => {
         const hotKeys2 = this.configs.value.filter(hk => !keys.includes(hk));
         this.configs.next(hotKeys2);
      };
   }

   public isHelpVisible(): boolean {
      return this.visibleHelp.value;
   }

   public getKeyNames(record: IHotKey): string[] {
      const key = record.hotKey;
      const result = [];

      if (key.ctrlKey) {
         result.push('Ctrl');
      }

      if (key.altKey) {
         result.push('Alt');
      }

      if (key.metaKey) {
         result.push('Cmd/Win');
      }

      if (key.shiftKey) {
         result.push('Shift');
      }

      result.push(this.forHuman(key.key ? key.key : key.code));

      return result;
   }

   private forHuman(key: string): string {
      return this.keySymbols.get(key) || key.replace(/^Key/, '');
   }

   private isMatched(e: KeyboardEvent, key: IKey): boolean {
      return Object.keys(key).every(field => e[field] === key[field]);
   }
}

export const globalHotKeys = new GlobalHotKeys();
