import { Observable, Observer } from 'rxjs';

import { json } from '../formatters';
import { isEmpty, toQuery } from '../helpers';

import { HttpMethod, HttpStatusCode, IApiError } from './models';

export interface RequestNamedParams<P, B> {
   body?: B;
   credentials: RequestCredentials;
   headers?: Record<string, string>;
   hooks?: ApiHooks;
   method: HttpMethod;
   onlyJson: boolean;
   params?: P;
   path: string;
}

type RequestListParams<P, B> = [HttpMethod, string, P?, B?, boolean?];

type RequestParams<P, B> = [RequestNamedParams<P, B>] | RequestListParams<P, B>;

type ApiHook = (init: RequestInit) => void;

export interface ApiHooks {
   before?: ApiHook;
   error?: ApiHook;
   ok?: ApiHook;
}

/**
 * У пользователей старых браузеров бывают ошибки.
 * Т.к. фича необязательная, буду для них возвращать заглушку.
 */
function getAbortController() {
   if ('AbortController' in window) {
      return new AbortController();
   }

   return {
      abort() {
         console.warn('Can not cancel request. Please, update your browser!');
      },
      signal: undefined,
   };
}

function parseErrorTextFromHtml(html: string): string {
   const finders = [/<pre>(.+?)<br>/, /<h1>(.+?)<\/h1>/, /<title>(.+?)<\/title>/];

   for (const finder of finders) {
      const match = html.match(finder);
      if (match) {
         return match[1];
      }
   }

   return html;
}

function getRequestParams<P, B>(args: RequestParams<P, B>): RequestNamedParams<P, B> {
   let method: HttpMethod;
   let path: string;
   let params: P | undefined;
   let body: B | undefined;
   let onlyJson: boolean | undefined;
   let headers: Record<string, string> | undefined;
   let credentials: RequestCredentials | undefined;
   let hooks: ApiHooks | undefined;

   if (typeof args[0] === 'string') {
      const [_method, _path, _params, _body, _onlyJson] = args as RequestListParams<P, B>;
      method = _method;
      path = _path;
      params = _params;
      body = _body;
      onlyJson = _onlyJson;
   } else {
      const {
         method: _method,
         path: _path,
         params: _params,
         body: _body,
         onlyJson: _onlyJson,
         credentials: _credentials,
         headers: _headers,
         hooks: _hooks,
      } = args[0];
      method = _method;
      path = _path;
      params = _params;
      body = _body;
      onlyJson = _onlyJson;
      credentials = _credentials;
      headers = _headers;
      hooks = _hooks;
   }

   return {
      method,
      path,
      params,
      body,
      onlyJson: onlyJson ?? true,
      credentials: credentials ?? 'include',
      headers,
      hooks,
   };
}

export abstract class BaseApi {
   constructor(protected apiPrefix: string | null, protected hooks: ApiHooks | null = null) {}

   protected abstract getCSRF(): string;

   protected abstract handleError(resp: Response, error: IApiError): void;

   protected getFetchParams(
      method: HttpMethod,
      body: string | undefined,
      headers: HeadersInit,
      credentials: RequestCredentials,
   ): RequestInit {
      return {
         body,
         credentials,
         headers,
         method,
      };
   }

   protected request<P, B, R>(
      method: HttpMethod,
      path: string,
      params?: P,
      body?: B,
      onlyJson?: boolean,
   ): Observable<R>;

   protected request<P, B, R>(params: RequestNamedParams<P, B>): Observable<R>;

   protected request<P, B, R>(...args: RequestParams<P, B>): Observable<R> {
      const {
         method,
         path,
         params,
         body,
         onlyJson,
         credentials,
         headers: customHeaders,
         hooks: customHooks,
      } = getRequestParams<P, B>(args);

      const url = this.buildUrl(path, params);
      const serializedBody = body === undefined ? undefined : json(body, 0);

      const hooks = { ...(this.hooks ?? {}), ...(customHooks ?? {}) };

      return new Observable((observer: Observer<R>) => {
         const abortController = getAbortController();

         const headers: HeadersInit = {};

         if (method !== HttpMethod.GET) {
            headers['Content-Type'] = 'application/json';
            if (this.getCSRF()) {
               headers['X-CSRF-Token'] = this.getCSRF();
            }
         }

         if (customHeaders) {
            Object.assign(headers, customHeaders);
         }

         const requestInit: RequestInit = {
            ...this.getFetchParams(method, serializedBody, headers, credentials),
            signal: abortController.signal,
         };

         if (hooks.before) {
            hooks.before(requestInit);
         }

         fetch(url, requestInit)
            .then(resp => this.parseResponse<R>(resp, onlyJson ?? true))
            .then(
               parsed => {
                  if (hooks.ok) {
                     hooks.ok(requestInit);
                  }
                  observer.next(parsed);
                  observer.complete();
               },
               error => {
                  if (hooks.error) {
                     hooks.error(requestInit);
                  }
                  observer.error(error);
               },
            );

         // unsubscribe function
         return () => abortController.abort();
      });
   }

   // eslint-disable-next-line @typescript-eslint/no-unused-vars
   protected parseError(body: any, resp: Response): any {
      return body;
   }

   private buildUrl<P>(path: string, params?: P): string {
      const isAbsolute = this.apiPrefix === null || path.startsWith('/');
      const prefixSeparator = this.apiPrefix === null || this.apiPrefix.endsWith('/') ? '' : '/';

      const pathWithPrefix = isAbsolute ? path : `${this.apiPrefix}${path ? `${prefixSeparator}${path}` : ''}`;

      const query = isEmpty(params) ? '' : `?${toQuery(params)}`;

      return `${pathWithPrefix}${query}`;
   }

   private isStatusOk(code: number) {
      return code !== HttpStatusCode.NoConnection && code < HttpStatusCode.BadRequest;
   }

   private isStatusWithBody(code: number): boolean {
      // 204 or 30x
      if (
         code === HttpStatusCode.NoContent ||
         (code >= HttpStatusCode.MultipleChoices && code < HttpStatusCode.BadRequest)
      ) {
         return false;
      }

      // 20x, 40x, 50x
      return true;
   }

   private async parseResponse<R>(resp: Response, onlyJson: boolean): Promise<R> {
      const contentType = resp.headers.get('Content-Type') || '';

      if (resp.status === HttpStatusCode.NoConnection) {
         const offlineError = { errorMessage: 'API is unavailable, check internet connection' };
         this.handleError(resp, offlineError);

         return Promise.reject(offlineError);
      }

      const isBodyExpected = onlyJson && this.isStatusWithBody(resp.status);
      const rawData = await resp.text();
      const hasBody = rawData.length > 0;

      if (isBodyExpected && !hasBody) {
         const emptyBodyError = { errorMessage: 'Body is expected' };
         this.handleError(resp, emptyBodyError);

         return Promise.reject(emptyBodyError);
      }

      // В теории ошибки всегда в JSON
      const isJson = this.isStatusOk(resp.status)
         ? hasBody && onlyJson
         : hasBody && contentType.includes('application/json');

      let result: any = isJson ? undefined : rawData;
      let parsedOk = isJson ? undefined : true;

      if (isJson) {
         try {
            result = JSON.parse(rawData);
            parsedOk = true;
         } catch (e) {
            result = { errorMessage: parseErrorTextFromHtml(rawData) };
            parsedOk = false;
         }
      }

      // success
      if (parsedOk && this.isStatusOk(resp.status)) {
         return result;
      }

      // error
      this.handleError(resp, result);

      return Promise.reject(this.parseError(result, resp));
   }
}
