import { CacheStore } from './models';

export class PersistentStore implements CacheStore {
   private innerStore = new Map<string, { ts: number; value: any }>();

   /**
    *
    * @param ttl Время жизни в секундах
    * @param prefix Префикс для localStorage
    */
   constructor(private ttl = 0, private prefix: string = 'ApiCache') {}

   public get(key: string): any {
      if (this.innerStore.has(key)) {
         const item = this.innerStore.get(key)!;
         if (this.isActual(item.ts)) {
            return item.value;
         }
         this.innerStore.delete(key); // side-effect
      }

      const storageKey = this.createStorageKey(key);
      if (storageKey in localStorage) {
         const item = JSON.parse(localStorage.getItem(storageKey)!);
         if (this.isActual(item.ts)) {
            return item.value;
         }
         localStorage.removeItem(storageKey); // side-effect
      }

      return undefined;
   }

   public has(key: string): boolean {
      if (this.innerStore.has(key)) {
         const item = this.innerStore.get(key)!;

         return this.isActual(item.ts);
      }

      const storageKey = this.createStorageKey(key);
      if (storageKey in localStorage) {
         const item = JSON.parse(localStorage.getItem(storageKey)!);

         return this.isActual(item.ts);
      }

      return false;
   }

   public set(key: string, value: any): void {
      const item = { ts: Date.now(), value };

      const storageKey = this.createStorageKey(key);

      this.innerStore.set(key, item); // side-effect
      localStorage.setItem(storageKey, JSON.stringify(item)); // side-effect
   }

   private createStorageKey(key: string): string {
      return `${this.prefix}/${key}`;
   }

   private isActual(ts: number): boolean {
      if (this.ttl === 0) {
         return true;
      }

      return Date.now() / 10e6 - ts <= this.ttl;
   }
}
