import { action } from '@storybook/addon-actions';
import { Meta, Story } from '@storybook/react';
import { Button, DialogProps } from '@yandex-cloud/uikit';
import * as React from 'react';
import { useCallback } from 'react';
import { takeUntil } from 'rxjs/operators';

import { modalsSupportDecorator } from '../../_storybook';
import { useDismounted } from '../../react_hooks';
import { modalService } from '../modalService';
import { IModalProps } from '../models';

interface IProps extends IModalProps<string> {
   text: string;
}

const NestedModalContent = ({ ok, cancel, text }: IProps) => (
   <>
      <h2>{text}</h2>

      <Button onClick={() => cancel('close from nested')}>Cancel</Button>

      <Button view={'action'} onClick={() => ok('ok in nested')}>
         Ok
      </Button>
   </>
);

const ModalContent = ({ ok, cancel }: IProps) => {
   const dismounted = useDismounted();

   const close = () => cancel('Close from modal programmatically');

   const openNested = useCallback(() => {
      modalService
         .open(NestedModalContent, { text: 'Nested modal without ModalLayout' })
         .pipe(takeUntil(dismounted))
         .subscribe(action('onNestedModalSuccess'), action('onNestedModalFailure'));
   }, [dismounted]);

   return (
      <>
         <Button onClick={openNested}>Open nested modal without layout</Button>

         <Button view={'action'} onClick={() => ok('ok')}>
            Ok
         </Button>

         <Button onClick={close}>Cancel</Button>
      </>
   );
};

interface StoryArgs {
   canClose: boolean;
   hasButtonClose: boolean;
   size?: DialogProps['size'];
   text: string;
}

const Template: Story<StoryArgs> = ({ canClose, hasButtonClose, text, size }) => {
   const dismounted = useDismounted();

   const open = () => {
      const s = modalService
         .openWithLayout(
            {
               onDismiss: (...args: any[]) => {
                  s.unsubscribe();

                  action('onLayoutDismiss')(...args);
               },
               onOk: action('onLayoutOk'),
               title: text,
            },
            ModalContent,
            { text },
            { closeOnOutsideClick: canClose, hasButtonClose, dialogProps: { size } },
         )
         .pipe(takeUntil(dismounted))
         .subscribe(action('onModalSuccess'), action('onModalFailure'));
   };

   return (
      <Button view={'action'} onClick={open}>
         Open with layout
      </Button>
   );
};

export const CanClose = Template.bind({});
CanClose.args = {
   canClose: true,
   hasButtonClose: true,
   text: 'This is closable modal',
};

export const CanNotClose = Template.bind({});
CanNotClose.args = {
   canClose: false,
   hasButtonClose: false,
   text: 'This is not closable modal',
};

export const SmallSize = Template.bind({});
SmallSize.args = {
   size: 's',
   text: 'some text',
};

export const MediumSize = Template.bind({});
MediumSize.args = {
   size: 'm',
   text: 'some text',
};

export const LargeSize = Template.bind({});
LargeSize.args = {
   size: 'l',
   text: 'some text',
};

export default {
   decorators: [modalsSupportDecorator],
   title: 'modals/modalService',
} as Meta;
