import { Dialog } from '@yandex-cloud/uikit';
import * as React from 'react';
import { ReactElement, SyntheticEvent, useEffect, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { getUniqueId } from '../../helpers';
import { useDismounted } from '../../react_hooks';
import { modalService } from '../modalService';
import { DISMISS_REASON, IModalConfig } from '../models';

import { ModalLayout } from './ModalLayout';

interface OpenedModal {
   config: IModalConfig<any, any>;
   element: ReactElement;

   handleOutsideClick(e: SyntheticEvent | MouseEvent | KeyboardEvent, opts: { isOutsideClick: boolean }): void;
}

export const ModalContainer = React.memo(() => {
   // hooks
   const dismounted = useDismounted();
   const [, setRenderCounter] = useState(0);
   const openedModals = useMemo(() => new Map<number, OpenedModal>(), []);

   // subscription
   useEffect(() => {
      const s = modalService.modals.pipe(takeUntil(dismounted)).subscribe(modalConfig => {
         const id = getUniqueId('modals');

         /**
          * Close modal
          */
         const clear = () => {
            openedModals.delete(id);
            setRenderCounter(c => c + 1); // force rerender
         };

         // call clear when subscription will be unsubscribed
         modalConfig.observer.add(clear);

         // Декорируем методы cancel, ok
         const props = {
            ...modalConfig.props,
            cancel: (r: any) => {
               modalConfig.observer.error(r || DISMISS_REASON);
            },
            ok: (r: any) => {
               modalConfig.observer.next(r);
               modalConfig.observer.complete();
            },
         };

         openedModals.set(id, {
            config: modalConfig,
            element: React.createElement(modalConfig.component, props),
            handleOutsideClick(e, { isOutsideClick }) {
               if (modalConfig.options.closeOnOutsideClick || !isOutsideClick) {
                  props.cancel(DISMISS_REASON);
                  clear();
               }
            },
         });
         setRenderCounter(c => c + 1); // force rerender
      });

      return () => s.unsubscribe();
   }, [dismounted, openedModals]);

   // render
   return (
      <>
         {Array.from(openedModals.entries()).map(([id, { config, element, handleOutsideClick }]) => (
            <Dialog
               key={id.toString()}
               open={true}
               hasCloseButton={config.options.hasButtonClose}
               onOutsideClick={e => handleOutsideClick(e, { isOutsideClick: true })}
               onEscapeKeyDown={e => handleOutsideClick(e, { isOutsideClick: false })}
               onClose={(e, r) => handleOutsideClick(e, { isOutsideClick: r === 'outsideClick' })}
               {...config.options.dialogProps}
            >
               {config.withLayout ? <ModalLayout {...config.layoutProps!}>{element}</ModalLayout> : element}
            </Dialog>
         ))}
      </>
   );
});

ModalContainer.displayName = 'ModalContainer';
