import { MutableRefObject, useCallback, useState } from 'react';

import { debounceFunc } from '../helpers';

import { useWindowChangeDimensions } from './useWindowChangeDimensions';

/**
 * Позволяет подстаивать высоту popup под размер экрана
 * (отступ от края экрана 40px, минимальная высота 200px).
 *
 * Возвращает тюпл: [ширина popup, его максимальная высота].
 *
 * @param switcherRef элемент, относительно которого нужно подстроить
 */
export function useDropdownDimensions(switcherRef: MutableRefObject<HTMLElement | undefined>) {
   const [width, setWidth] = useState(200);
   const [maxHeight, setMaxHeight] = useState(400);

   // eslint-disable-next-line react-hooks/exhaustive-deps
   const updateDimensions = useCallback(
      debounceFunc(() => {
         const bottomEdgeMargin = 40;
         const minHeight = 200;

         if (!switcherRef.current) {
            return;
         }

         const rect = switcherRef.current.getBoundingClientRect() as DOMRect;

         // width
         const newMinWidth = rect.width;
         if (newMinWidth !== width) {
            setWidth(newMinWidth);
         }

         // max height
         const newMaxHeight = Math.max(minHeight, window.innerHeight - rect.y - rect.height - bottomEdgeMargin);
         if (newMaxHeight !== maxHeight) {
            setMaxHeight(newMaxHeight);
         }
      }, 250),
      [switcherRef.current, width, maxHeight],
   );

   useWindowChangeDimensions(updateDimensions);

   return [width, maxHeight] as [number, number];
}
