import { Dispatch, SetStateAction, useCallback, useState } from 'react';

type parserOptions<T> =
   | {
        raw: true;
     }
   | {
        raw: false;
        serializer: (value: T) => string;
        deserializer: (value: string) => T;
     };

type AnyFunction = (...args: any[]) => any;

// ported from https://github.com/streamich/react-use/blob/master/src/useLocalStorage.ts
export function useLocalStorage<T>(
   key: string,
   initialValue?: T,
   options?: parserOptions<T>,
): [T | undefined, Dispatch<SetStateAction<T | undefined>>, () => void] {
   if (!key) {
      throw new Error('useLocalStorage key may not be falsy');
   }

   const rawDeserializer = useCallback((value: any) => value, []);

   const serializer = options ? (options.raw ? String : options.serializer) : JSON.stringify;
   const deserializer = options ? (options.raw ? rawDeserializer : options.deserializer) : JSON.parse;

   // eslint-disable-next-line react-hooks/rules-of-hooks
   const [state, setState] = useState<T | undefined>(() => {
      try {
         const localStorageValue = localStorage.getItem(key);
         if (localStorageValue !== null) {
            return deserializer(localStorageValue);
         }
         if (initialValue) {
            localStorage.setItem(key, serializer(initialValue));
         }

         return initialValue;
      } catch {
         // If user is in private mode or has storage restriction
         // localStorage can throw. JSON.parse and JSON.stringify
         // can throw, too.
         return initialValue;
      }
   });

   // eslint-disable-next-line react-hooks/rules-of-hooks
   const set: Dispatch<SetStateAction<T | undefined>> = useCallback(
      valOrFunc => {
         try {
            const newState = typeof valOrFunc === 'function' ? (valOrFunc as AnyFunction)(state) : valOrFunc;
            if (typeof newState === 'undefined') return;
            let value: string;

            if (options)
               if (options.raw)
                  if (typeof newState === 'string') value = newState;
                  else value = JSON.stringify(newState);
               else if (options.serializer) value = options.serializer(newState);
               else value = JSON.stringify(newState);
            else value = JSON.stringify(newState);

            localStorage.setItem(key, value);
            setState(deserializer(value));
         } catch {
            // If user is in private mode or has storage restriction
            // localStorage can throw. Also JSON.stringify can throw.
         }
      },
      [deserializer, key, options, state],
   );

   // eslint-disable-next-line react-hooks/rules-of-hooks
   const remove = useCallback(() => {
      try {
         localStorage.removeItem(key);
         setState(undefined);
      } catch {
         // If user is in private mode or has storage restriction
         // localStorage can throw.
      }
   }, [key, setState]);

   return [state, set, remove];
}
