import { useCallback, useEffect, useRef } from 'react';

function preventEventIfFocused(e: WheelEvent) {
   const target = e.target as HTMLInputElement;
   // Обращение к DOM, увы, это единственный известный мне способ узнать, в фокусе ли элемент.
   if (target === document.activeElement) {
      e.preventDefault();
   }
}

/**
 * Хук для предотвращения скролла внутри текстовых полей.
 *
 * Актуально для input[type=number], где по скроллу меняется значение.
 *
 * #DEPLOY-4772
 *
 * @example
 *    const { setInputRef } = usePreventScrollForInputs();
 *    <TextInput controlRef={setInputRef} ... />
 */
export function usePreventScrollForInputs() {
   const inputRef = useRef<HTMLInputElement>();

   const setInputRef = useCallback((input: HTMLInputElement) => {
      inputRef.current = input;
   }, []);

   useEffect(() => {
      inputRef.current?.addEventListener('wheel', preventEventIfFocused);

      return () => {
         inputRef.current?.removeEventListener('wheel', preventEventIfFocused);
      };
   }, []);

   return { setInputRef };
}
