import * as React from 'react';
import { useContext, useEffect, useState } from 'react';

import { SystemUsersContext } from '../../_models';
import { classNames } from '../../formatters';
import { isEmpty } from '../../helpers';

import styles from './Avatar.module.css';
import fallbackUserAvatar from './fallback-user.png';
import userGroupAvatar from './user-group.png';

const AVATAR_HOST = 'https://center.yandex-team.ru';

const checkedAvatars = new Map<string, boolean>();

function isAvatarExist(login: string): Promise<boolean> {
   if (checkedAvatars.has(login)) {
      return Promise.resolve(checkedAvatars.get(login)!);
   }

   return new Promise<boolean>(resolve => {
      const img = new Image();
      img.src = `${AVATAR_HOST}/api/v1/user/${login}/avatar/100.jpg`;
      img.onload = () => resolve(true);
      img.onerror = () => resolve(false);
   }).then(isExist => {
      checkedAvatars.set(login, isExist);

      return isExist;
   });
}

interface IProps {
   big?: boolean;
   login: string | 'service' | null;
}

/**
 * Компонент для отображения аватарок сотрудников/роботов/служб
 */
export const Avatar = React.memo(({ login, big = false }: IProps) => {
   const systemUsers = useContext(SystemUsersContext);
   const systemUser = login ? systemUsers.get(login) : null;
   const [exist, setExists] = useState<boolean>(false);

   useEffect(() => {
      if (login === 'service' || systemUser) {
         return;
      }

      if (typeof login !== 'string' || isEmpty(login)) {
         setExists(false);

         return;
      }

      isAvatarExist(login).then(setExists);
   }, [login, systemUser]);

   const cls = classNames(styles.avatar, {
      [styles.systemUser]: Boolean(systemUser),
      [styles.big]: big,
      [styles.small]: !big,
   });

   if (login === 'service') {
      return (
         <div className={cls} style={{ backgroundImage: `url(${userGroupAvatar})`, backgroundSize: '100% 100%' }} />
      );
   }

   if (systemUser) {
      return <div className={cls} style={{ backgroundImage: `url(${systemUser.avatar})` }} />;
   }

   if (!exist) {
      return (
         <div className={cls} style={{ backgroundImage: `url(${fallbackUserAvatar})`, backgroundSize: '100% 100%' }} />
      );
   }

   if (!login) {
      return <div className={cls} />;
   }

   const regular = `${AVATAR_HOST}/api/v1/user/${login}/avatar/42.jpg`;
   const retina = `${AVATAR_HOST}/api/v1/user/${login}/avatar/100.jpg`;

   return (
      <div className={classNames(styles.userPic, cls)}>
         <img src={regular} srcSet={`${regular} 1x, ${retina} 2x`} alt={login} />
      </div>
   );
});

Avatar.displayName = 'Avatar';
