import { Dispatch, ReducerAction } from 'react';

import { getSetDifference, isEmpty, isEqual, toggleSetItem } from '../../helpers';

export type ID = number | string;

export enum ActionType {
   UpdateSelectedAll = 'UpdateSelectedAll',
   Clear = 'Clear',
   DeselectItems = 'DeselectItems',
   SelectItems = 'SelectItems',
   SetAllItems = 'SetAllItems',
   ToggleAll = 'ToggleAll',
   ToggleItem = 'ToggleItem',
}

interface IActionUpdateSelectedAll {
   type: ActionType.UpdateSelectedAll;
}

interface IActionClear {
   type: ActionType.Clear;
}

interface IActionDeselectItems {
   items: Set<ID>;
   type: ActionType.DeselectItems;
}

interface IActionSelectItems {
   items: Set<ID>;
   type: ActionType.SelectItems;
}

interface IActionSetAllItems {
   items: ID[];
   type: ActionType.SetAllItems;
}

interface IActionToggleAll {
   type: ActionType.ToggleAll;
}

interface IActionToggleItem {
   item: ID;
   shiftPressed: boolean;
   type: ActionType.ToggleItem;
}

export type Action =
   | IActionUpdateSelectedAll
   | IActionClear
   | IActionDeselectItems
   | IActionSelectItems
   | IActionSetAllItems
   | IActionToggleAll
   | IActionToggleItem;

export const initialState = {
   allItems: [] as ID[],
   isAllSelected: false as boolean | null,
   lastToggledItem: null as ID | null,
   selected: new Set<ID>(),
};

export type State = Readonly<typeof initialState>;

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.UpdateSelectedAll: {
         let isAllSelected = null;

         const hasItems = !isEmpty(state.allItems);
         const hasSelection = !isEmpty(state.selected);

         if (!hasItems || !hasSelection) {
            isAllSelected = false;
         } else if (isEqual(new Set(state.allItems), state.selected)) {
            isAllSelected = true;
         }

         return { ...state, isAllSelected };
      }

      case ActionType.Clear: {
         const newState = { ...state, selected: new Set<string>() };

         return reducer(newState, { type: ActionType.UpdateSelectedAll });
      }

      case ActionType.DeselectItems: {
         const selected = new Set(state.selected);
         action.items.forEach(item => selected.delete(item));

         return { ...state, selected };
      }

      case ActionType.SelectItems: {
         const selected = new Set(state.selected);
         action.items.forEach(item => selected.add(item));

         return { ...state, selected };
      }

      case ActionType.SetAllItems: {
         const selected = new Set(state.selected);

         const { removed } = getSetDifference(new Set(state.allItems), new Set(action.items));
         removed.forEach(item => selected.delete(item));

         const newState: State = {
            ...state,
            allItems: action.items,
            selected,
         };

         return reducer(newState, { type: ActionType.UpdateSelectedAll });
      }

      case ActionType.ToggleAll: {
         let newState = state;
         if (state.isAllSelected) {
            newState = { ...newState, selected: new Set() };
         } else {
            newState = { ...newState, selected: new Set(state.allItems) };
         }

         return reducer(newState, { type: ActionType.UpdateSelectedAll });
      }

      case ActionType.ToggleItem: {
         let newState: State = {
            ...state,
            lastToggledItem: action.item,
            selected: toggleSetItem(state.selected, action.item),
         };

         // handle toggle with Shift pressed
         if (action.shiftPressed && state.lastToggledItem !== null) {
            const lastIndex = state.allItems.indexOf(state.lastToggledItem);
            const currentIndex = state.allItems.indexOf(action.item);
            const min = Math.min(lastIndex, currentIndex);
            const max = Math.max(lastIndex, currentIndex);

            const items = new Set(state.allItems.slice(min, max));
            items.add(state.lastToggledItem);
            items.add(action.item);

            const wasSelected = state.selected.has(action.item);
            if (wasSelected) {
               newState = reducer(newState, { type: ActionType.DeselectItems, items });
            } else {
               newState = reducer(newState, { type: ActionType.SelectItems, items });
            }
         }

         return reducer(newState, { type: ActionType.UpdateSelectedAll });
      }

      default: {
         return state;
      }
   }
};

export type Dispatcher = Dispatch<ReducerAction<typeof reducer>>;
