import { Meta } from '@storybook/react';
import { Button, Checkbox } from '@yandex-cloud/uikit';
import React, { useCallback, useMemo, useState } from 'react';

import { IListOption } from '../../form_inputs';
import { getUniqueId } from '../../helpers';

import { CheckboxSelection } from './CheckboxSelection';
import { ID } from './CheckboxSelection.state';
import { CheckboxSelectionItem } from './CheckboxSelectionItem';

const initialItems: IListOption[] = [
   { value: 'item1', name: 'Item 1' },
   { value: 'item2', name: 'Item 2' },
   { value: 'item3', name: 'Item 3' },
   { value: 'item4', name: 'Item 4' },
];

const Story = () => {
   const [items, setItems] = useState(initialItems);
   const [selection, setSelection] = useState(new Set<ID>(['item1']));

   const addRow = useCallback(() => {
      const id = getUniqueId('item') + initialItems.length;
      setItems(i => [...i, { value: `item${id}`, name: `Item ${id}` }]);
   }, []);

   const removeRow = useCallback((id: string) => setItems(i => i.filter(item => item.value !== id)), []);

   const allIds = useMemo(() => items.map(i => i.value), [items]);

   return (
      <CheckboxSelection allIds={allIds} value={selection} onChange={setSelection}>
         {({ isAllSelected, toggleAll, clear }) => (
            <>
               <Button onClick={addRow}>Add row</Button>

               <Button onClick={clear}>Clear selection</Button>

               <table style={{ width: 200 }}>
                  <thead>
                     <tr>
                        <th style={{ width: 25 }}>
                           <Checkbox
                              checked={isAllSelected !== false}
                              onChange={toggleAll}
                              indeterminate={isAllSelected === null}
                           />
                        </th>
                        <th>Name</th>
                        <th>Actions</th>
                     </tr>
                  </thead>
                  <tbody>
                     {items.map(item => (
                        <tr key={item.value}>
                           <td>
                              <CheckboxSelectionItem key={item.value} id={item.value}>
                                 {({ selected, toggle }) => <Checkbox checked={selected} onChange={toggle} />}
                              </CheckboxSelectionItem>
                           </td>
                           <td>{item.name}</td>
                           <td>
                              {/* tslint:disable-next-line:jsx-no-lambda */}
                              <Button size={'s'} onClick={() => removeRow(item.value)}>
                                 remove
                              </Button>
                           </td>
                        </tr>
                     ))}
                  </tbody>
               </table>
            </>
         )}
      </CheckboxSelection>
   );
};

export const Default = () => <Story />;

export default {
   title: 'small_components/CheckboxSelection',
} as Meta;
