import { ActionType, initialState, reducer, State } from './CheckboxSelection.state';

describe('small_components|CheckboxSelection', () => {
   describe('reducer tests', () => {
      const add4Items = (s: State) =>
         reducer(s, {
            items: ['item0', 'item1', 'item2', 'item3'],
            type: ActionType.SetAllItems,
         });

      let state: State;
      let stateWithItems: State;

      beforeAll(() => {
         state = initialState;
         stateWithItems = add4Items(initialState);
      });

      describe('handle adding items', () => {
         it('should add to allItems', () => {
            const newState = add4Items(state);

            expect(newState.allItems).toEqual(['item0', 'item1', 'item2', 'item3']);
            expect(newState.isAllSelected).toBe(false);
         });

         it('should update isAllSelected', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleAll });
            expect(newState.isAllSelected).toBe(true);

            newState = reducer(newState, {
               items: ['item0', 'item1', 'item2', 'item3', 'item4'],
               type: ActionType.SetAllItems,
            });
            expect(newState.isAllSelected).toBe(null);
         });
      });

      describe('handle removing items', () => {
         it('should remove item from allItems', () => {
            const newState = reducer(stateWithItems, {
               items: ['item0', 'item1', 'item3'],
               type: ActionType.SetAllItems,
            });

            expect(newState.allItems).toEqual(['item0', 'item1', 'item3']);
            expect(newState.isAllSelected).toBe(false);
         });

         it('should update isAllSelected on removing', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleAll });
            newState = reducer(newState, { type: ActionType.SetAllItems, items: ['item0', 'item1', 'item3'] });

            expect(newState.selected).toEqual(new Set(['item0', 'item1', 'item3']));
            expect(newState.isAllSelected).toBe(true);
         });

         it('should update isAllSelected on removing selected', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleItem, item: 'item2', shiftPressed: false });
            expect(newState.isAllSelected).toBe(null);

            newState = reducer(newState, { type: ActionType.SetAllItems, items: ['item0', 'item1', 'item3'] });

            expect(newState.selected).toEqual(new Set());
            expect(newState.isAllSelected).toBe(false);
         });
      });

      describe('toggleItem', () => {
         it('should toggle item in selected', () => {
            // select
            let newState = reducer(stateWithItems, { type: ActionType.ToggleItem, item: 'item1', shiftPressed: false });

            expect(newState.selected).toEqual(new Set(['item1']));
            expect(newState.lastToggledItem).toBe('item1');
            // unselect
            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item1', shiftPressed: false });

            expect(newState.selected).toEqual(new Set());
            expect(newState.lastToggledItem).toBe('item1');
         });

         it('should update isAllSelected', () => {
            let newState = stateWithItems;
            expect(newState.isAllSelected).toBe(false);

            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item0', shiftPressed: false });
            expect(newState.isAllSelected).toBe(null);

            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item1', shiftPressed: false });
            expect(newState.isAllSelected).toBe(null);

            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item2', shiftPressed: false });
            expect(newState.isAllSelected).toBe(null);

            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item3', shiftPressed: false });
            expect(newState.isAllSelected).toBe(true);
         });

         it('should select range if shift key is pressed', () => {
            let newState = stateWithItems;
            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item0', shiftPressed: false });
            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item2', shiftPressed: true });

            expect(newState.selected).toEqual(new Set(['item0', 'item1', 'item2']));
         });

         it('should deselect range if shift key is pressed', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleAll });
            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item0', shiftPressed: false });
            newState = reducer(newState, { type: ActionType.ToggleItem, item: 'item2', shiftPressed: true });

            expect(newState.selected).toEqual(new Set(['item3']));
         });
      });

      describe('toggleAll', () => {
         it('should select all if has no selection', () => {
            const newState = reducer(stateWithItems, { type: ActionType.ToggleAll });

            expect(newState.selected).toEqual(new Set(['item0', 'item1', 'item2', 'item3']));
            expect(newState.isAllSelected).toBe(true);
         });

         it('should select all if has some selection, but not all', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleItem, item: 'item0', shiftPressed: false });
            expect(newState.isAllSelected).toBe(null);

            newState = reducer(newState, { type: ActionType.ToggleAll });

            expect(newState.selected).toEqual(new Set(['item0', 'item1', 'item2', 'item3']));
            expect(newState.isAllSelected).toBe(true);
         });

         it('should clear selection if has all selected', () => {
            let newState = reducer(stateWithItems, { type: ActionType.ToggleAll });
            newState = reducer(newState, { type: ActionType.ToggleAll });

            expect(newState.selected).toEqual(new Set([]));
            expect(newState.isAllSelected).toBe(false);
         });
      });

      it('should clear selection', () => {
         let newState = reducer(stateWithItems, { type: ActionType.ToggleItem, item: 'item0', shiftPressed: false });
         newState = reducer(newState, { type: ActionType.Clear });

         expect(newState.selected).toEqual(new Set());
         expect(newState.isAllSelected).toBe(false);
      });

      it('should return unchanged state for unknown action', () => {
         const newState = reducer(stateWithItems, { type: 'UnknownAction' as any });

         expect(newState).toBe(stateWithItems);
      });
   });
});
