import React, { ReactNode, useEffect, useMemo, useReducer, useRef } from 'react';

import { IPureValueProps } from '../../_models';

import { ActionType, Dispatcher, ID, initialState, reducer, State } from './CheckboxSelection.state';
import { CheckboxSelectionContext } from './context';

interface Args {
   /**
    * Is "common" checkbox selected marker.
    *
    * If list has selection, but not all elements selected, this field will be `null`
    */
   isAllSelected: boolean | null;

   /**
    * Method for toggle "common" checkbox.
    */
   toggleAll(): void;

   /**
    * Optional method to clear the selection
    */
   clear(): void;
}

interface Props extends IPureValueProps<Set<ID>> {
   allIds: ID[];

   children(p: Args): ReactNode;
}

export const CheckboxSelection: React.FC<Props> = React.memo(({ allIds, value, onChange, children }) => {
   const firstRunRef = useRef(true);
   const [state, dispatch] = useReducer(reducer, { ...initialState, selected: value });

   useEffect(() => {
      dispatch({ type: ActionType.SetAllItems, items: allIds });
   }, [allIds]);

   useEffect(() => {
      if (!firstRunRef.current) {
         onChange(state.selected);
      } else {
         firstRunRef.current = false;
      }
   }, [onChange, state.selected]);

   const args: Args = useMemo(
      () => ({
         clear: () => dispatch({ type: ActionType.Clear }),
         isAllSelected: state.isAllSelected,
         toggleAll: () => dispatch({ type: ActionType.ToggleAll }),
      }),
      [state.isAllSelected],
   );

   const contextValue: [State, Dispatcher] = useMemo(() => [state, dispatch], [state]);

   return <CheckboxSelectionContext.Provider value={contextValue}>{children(args)}</CheckboxSelectionContext.Provider>;
});

CheckboxSelection.displayName = 'CheckboxSelection';
