import * as React from 'react';
import { useCallback, useEffect, useRef, useState } from 'react';

import { classNames } from '../../formatters';
import { copyToClipboard } from '../../helpers';

import styles from './CopyableText.module.css';

interface IProps {
   className?: string;
   text: string;
}

export const CopyableText = React.memo(({ text, className }: IProps) => {
   // hooks
   const timer = useRef<number>();
   const [justCopied, setJustCopied] = useState(false);

   // effects
   useEffect(() => () => window.clearTimeout(timer.current), []);

   // handlers
   const handleCopy = useCallback(() => {
      copyToClipboard(text);
      setJustCopied(true);
      timer.current = window.setTimeout(() => setJustCopied(false), 1000);
   }, [text]);

   // render
   // noinspection LongLine
   return (
      <span title={'Click to copy'} onClick={handleCopy} className={styles.iconWrapper} role={'button'} tabIndex={0}>
         <svg width={'16'} height={'16'} xmlns={'http://www.w3.org/2000/svg'}>
            <g className={classNames(styles.icon, className, { [styles.copied]: justCopied })}>
               <path
                  d={
                     'M4 6v6h6v-1.973H5.991V6H4zm0-1h1l.694 5.354L11 10.013V12a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V6a1 1 0 0 1 1-1z'
                  }
               />
               <path d={'M6 4v6h6V4H6zm0-1h6a1 1 0 0 1 1 1v6a1 1 0 0 1-1 1H6a1 1 0 0 1-1-1V4a1 1 0 0 1 1-1z'} />
            </g>
         </svg>
      </span>
   );
});

CopyableText.displayName = 'CopyableText';
