import { faAngleDown } from '@fortawesome/free-solid-svg-icons/faAngleDown';
import { faAngleUp } from '@fortawesome/free-solid-svg-icons/faAngleUp';
import { Button, Popup } from '@yandex-cloud/uikit';
import React, {
   forwardRef,
   ReactNode,
   Ref,
   SyntheticEvent,
   useCallback,
   useImperativeHandle,
   useRef,
   useState,
} from 'react';

import { FontAwesomeDataUICommonIcon } from '../../data-ui-common-extra';
import { classNames } from '../../formatters';
import { useDropdownDimensions } from '../../react_hooks';

import classes from './Dropdown.module.css';

export interface DropdownRef {
   close(): void;
}

interface Props {
   children: ReactNode;
   className?: string;
   disabled?: boolean;
   limitPopupWidth?: boolean;
   switcher: ReactNode;
   testE2e?: string;

   onToggle?(visible: boolean): void;
}

export const Dropdown = forwardRef(
   (
      { className, disabled = false, limitPopupWidth = true, children, switcher, testE2e, onToggle }: Props,
      ref: Ref<DropdownRef>,
   ) => {
      // hooks
      const [visible, setVisible] = useState(false);
      const switcherRef = useRef<HTMLDivElement>(null);
      const [width, maxHeight] = useDropdownDimensions(switcherRef as any);

      // handlers
      const toggle = useCallback(() => {
         setVisible(!visible);
         if (onToggle) {
            onToggle(!visible);
         }
      }, [onToggle, visible]);

      const close = useCallback(() => setVisible(false), []);

      const preventPropagation = useCallback((e: SyntheticEvent) => e.stopPropagation(), []);

      useImperativeHandle(ref, () => ({ close }), [close]);

      // render
      return (
         <>
            <div className={classNames('dropdown2', classes.list, className)} ref={switcherRef}>
               <Button className={classes.button} view={'raised'} onClick={toggle} disabled={disabled} qa={testE2e}>
                  {switcher}
                  <FontAwesomeDataUICommonIcon icon={visible ? faAngleUp : faAngleDown} size={14} />
               </Button>
            </div>

            <Popup anchorRef={switcherRef} placement={['bottom-start']} open={visible} onClose={close}>
               <div
                  style={{
                     [limitPopupWidth ? 'width' : 'minWidth']: width,
                     maxHeight,
                     overflowY: 'auto',
                  }}
                  onScroll={preventPropagation}
                  data-e2e={testE2e}
               >
                  {children}
               </div>
            </Popup>
         </>
      );
   },
);

Dropdown.displayName = 'Dropdown';
