import { Spin } from '@yandex-cloud/uikit';
import * as React from 'react';
import { useEffect, useState } from 'react';

import { classNames } from '../../formatters';
import { noop } from '../../helpers';
import { IModalProps, ModalLayout } from '../../modals';

import classes from './FeedbackModal.module.css';

interface FeedbackModalProps extends IModalProps<void> {
   formId: number;
   initialHeight?: number;
   title: string;
}

const DEFAULT_FORM_HEIGHT = 472; // Поля Types, Subject, Message, Files

function getFormUrl(formId: number) {
   return `https://forms.yandex-team.ru/surveys/${formId}/?iframe=1&url=${window.location.href}`;
}

export const FeedbackModal = React.memo(
   ({ formId, initialHeight = DEFAULT_FORM_HEIGHT, title }: FeedbackModalProps) => {
      const [isLoading, setIsLoading] = useState(true);
      const [height, setHeight] = useState(initialHeight);

      useEffect(() => {
         function receiveMessage(e: MessageEvent) {
            if (typeof e.data !== 'string' || !e.data.includes('iframe-height')) {
               return;
            }

            let data: { 'iframe-height': number };
            try {
               data = JSON.parse(e.data);
            } catch (_) {
               return;
            }

            const h = data['iframe-height'];
            if (h) {
               setHeight(h);
               setIsLoading(false);
            }
         }

         window.addEventListener('message', receiveMessage, false);

         return () => window.removeEventListener('message', receiveMessage);
      });

      return (
         <ModalLayout onDismiss={noop} title={title} showFooter={false}>
            <div className={classes.wrapper}>
               <div
                  style={{ height }}
                  className={classNames(classes.formLoader, {
                     [classes.loading]: isLoading,
                  })}
               >
                  <Spin className={classes.loader} size={'l'} />
               </div>

               <iframe
                  name={`ya-form-${formId}`}
                  src={getFormUrl(formId)}
                  style={{
                     border: 'none',
                     height,
                     width: '100%',
                  }}
                  title={title}
               />
            </div>
         </ModalLayout>
      );
   },
);

FeedbackModal.displayName = 'FeedbackModal';
