import { Popup, PopupPlacement } from '@yandex-cloud/uikit';
import * as React from 'react';
import { CSSProperties, ReactNode, RefObject, SyntheticEvent, useCallback, useEffect, useRef, useState } from 'react';

import { useDropdownDimensions } from '../../react_hooks';

interface IProps {
   children: ReactNode;
   direction?: 'free' | 'down';
   opened?: boolean;
   switcher: JSX.Element;

   tickRender?(opened: boolean): JSX.Element;

   onChangeOpened?(opened: boolean): void;
}

export const LazyDropdown = React.memo(
   ({ children, direction = 'down', onChangeOpened, opened, switcher, tickRender }: IProps) => {
      // hooks
      const switcherRef = useRef<HTMLSpanElement>(null);
      const [internalOpened, setInternalOpened] = useState(false);
      const [wasOpened, setWasOpened] = useState(false);

      // effects
      useEffect(() => {
         if (opened !== undefined) {
            setInternalOpened(opened);
            setWasOpened(true);
         }
      }, [opened]);

      useEffect(() => {
         if (onChangeOpened) {
            onChangeOpened(internalOpened);
         }
      }, [internalOpened, onChangeOpened]);

      // positioning
      const placement: PopupPlacement | undefined =
         direction === 'down' ? ['bottom-start', 'bottom', 'bottom-end'] : undefined;

      // eslint-disable-next-line react-hooks/rules-of-hooks
      const [, maxHeight] = direction === 'down' ? useDropdownDimensions(switcherRef as any) : [0, 0];

      const wrapperStyle: CSSProperties = direction === 'down' ? { maxHeight, overflowY: 'auto' } : {};

      // handlers
      const preventPropagation = useCallback((e: SyntheticEvent) => e.stopPropagation(), []);

      const toggle = useCallback(() => {
         const newOpened = !internalOpened;

         setInternalOpened(newOpened);
         setWasOpened(true);
      }, [internalOpened]);

      const close = useCallback(() => setInternalOpened(false), []);

      // render
      return (
         <>
            <span ref={switcherRef as RefObject<HTMLDivElement>} onClick={toggle} role={'button'} tabIndex={0}>
               {switcher}
               {tickRender ? tickRender(internalOpened) : null}
            </span>

            {wasOpened ? (
               <Popup
                  hasArrow={true}
                  onClose={close}
                  placement={placement}
                  open={internalOpened}
                  anchorRef={switcherRef}
               >
                  <div style={wrapperStyle} onScroll={preventPropagation}>
                     {children}
                  </div>
               </Popup>
            ) : null}
         </>
      );
   },
);

LazyDropdown.displayName = 'LazyDropdown';
