import { IconDefinition } from '@fortawesome/fontawesome-common-types';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { ControlGroupOption } from '@yandex-cloud/uikit';
import React, { ReactNode, SyntheticEvent, useCallback, useEffect } from 'react';

import { Keys, PER_PAGE_DEFAULT, PER_PAGE_VARIANTS } from '../../_models';
import {
   faArrowFromLeft,
   faArrowFromRight,
   faLongArrowLeft,
   faLongArrowRight,
} from '../../_styles/@fortawesome/pro-solid-svg-icons';
import { SimpleSelect } from '../../data-ui-common-extra';
// noinspection ES6PreferShortImport
import { EnumSwitcher } from '../../form_inputs/EnumSwitcher/EnumSwitcher';
import { globalHotKeys } from '../../hotkeys';

import classes from './Pagination.module.css';

enum PaginationButton {
   First = 'first',
   Previous = 'previous',
   Next = 'next',
   Last = 'last',
}

function createButton(config: {
   children?: ReactNode;
   disabled: boolean;
   hideTitle: boolean;
   icon?: IconDefinition;
   title?: string;
   value: string;
}): ControlGroupOption {
   return {
      disabled: config.disabled,
      value: config.value,
      content: (
         <span data-e2e={`Pagination:${config.value}`}>
            <span title={config.hideTitle ? undefined : config.title}>
               {config.icon ? <FontAwesomeIcon icon={config.icon} /> : null}
            </span>
            {config.children}
         </span>
      ),
   };
}

function getBounds(total: number, maxButtons: number, currentPage: number): [number, number] {
   let first = currentPage;
   let last = first + maxButtons - 1;

   // Конец диапазона, активная подсвеченная страница теперь не первая
   const diff = total - last;
   if (diff < 0) {
      first -= Math.abs(diff);
      if (first <= 0) {
         first = 1;
      }
      last = total;
   }

   return [first, last];
}

interface Props {
   hideSinglePage?: boolean;
   id?: string;
   maxButtons?: number;
   perPage?: number;
   perPageVariants?: number[];
   total: number;
   useGlobalHotkeys?: boolean;
   value: number;

   onChange(e: SyntheticEvent | null, page: number): void;

   onPerPageChange?(e: SyntheticEvent | null, perPage: number): void;
}

export const Pagination: React.FC<Props> = React.memo(
   ({
      hideSinglePage = false,
      id,
      maxButtons = 7,
      onChange,
      onPerPageChange,
      perPage = PER_PAGE_DEFAULT,
      perPageVariants = PER_PAGE_VARIANTS,
      total,
      useGlobalHotkeys,
      value,
   }) => {
      const handlePerPageChange = useCallback(
         (val: string) => {
            if (onPerPageChange) {
               onPerPageChange(null, parseInt(val, 10));
            }
         },
         [onPerPageChange],
      );

      const previousPage = value - 1;
      const nextPage = value < total ? value + 1 : value;
      const handlePageClick = useCallback(
         (v: PaginationButton | string) => {
            switch (v) {
               case PaginationButton.First: {
                  if (value > 1) {
                     onChange(null, 1);
                  }
                  break;
               }

               case PaginationButton.Previous: {
                  if (previousPage > 0) {
                     onChange(null, previousPage);
                  }
                  break;
               }

               case PaginationButton.Next: {
                  if (nextPage !== value) {
                     onChange(null, nextPage);
                  }
                  break;
               }

               case PaginationButton.Last: {
                  if (value < total) {
                     onChange(null, total);
                  }
                  break;
               }

               default: {
                  const page = parseInt(v, 10);
                  onChange(null, page);
                  break;
               }
            }
         },
         [nextPage, onChange, previousPage, total, value],
      );

      useEffect(() => {
         if (!useGlobalHotkeys) {
            return undefined;
         }

         return globalHotKeys.register(
            {
               action: () => handlePageClick(PaginationButton.First),
               help: 'First page',
               hotKey: { code: Keys.ArrowLeft, metaKey: true, ctrlKey: true, shiftKey: true },
            },
            {
               action: () => handlePageClick(PaginationButton.Previous),
               help: 'Previous page',
               hotKey: { code: Keys.ArrowLeft, metaKey: true, ctrlKey: true },
            },
            {
               action: () => handlePageClick(PaginationButton.Next),
               help: 'Next page',
               hotKey: { code: Keys.ArrowRight, metaKey: true, ctrlKey: true },
            },
            {
               action: () => handlePageClick(PaginationButton.Last),
               help: 'Last page',
               hotKey: { code: Keys.ArrowRight, metaKey: true, ctrlKey: true, shiftKey: true },
            },
         );
      }, [handlePageClick, useGlobalHotkeys]);

      // region render
      const firstButton = createButton({
         disabled: value === 1,
         hideTitle: !useGlobalHotkeys,
         icon: faArrowFromRight,
         title: 'Ctrl + Shift + Cmd/Win + ←',
         value: PaginationButton.First,
      });

      const prevButton = createButton({
         disabled: previousPage <= 0,
         hideTitle: !useGlobalHotkeys,
         icon: faLongArrowLeft,
         title: 'Ctrl + Cmd/Win + ←',
         value: PaginationButton.Previous,
      });

      const [firstPageButton, lastPageButton] = getBounds(total, maxButtons, value);
      const buttons: ControlGroupOption[] = [];
      for (let page = firstPageButton; page <= lastPageButton; page += 1) {
         buttons.push(
            createButton({
               children: <span className={classes.page}>{page}</span>,
               disabled: false,
               hideTitle: !useGlobalHotkeys,
               value: page.toString(10),
            }),
         );
      }

      const nextButton = createButton({
         disabled: nextPage === value,
         hideTitle: !useGlobalHotkeys,
         icon: faLongArrowRight,
         title: 'Ctrl + Cmd/Win + →',
         value: PaginationButton.Next,
      });
      const lastButton = createButton({
         children: total > 1 ? <span className={classes.total}>{total}</span> : null,
         disabled: value === total,
         hideTitle: !useGlobalHotkeys,
         icon: faArrowFromLeft,
         title: 'Ctrl + Shift + Cmd/Win + →',
         value: PaginationButton.Last,
      });

      if (hideSinglePage && buttons.length < 2) {
         return null;
      }

      return (
         <div className={classes.pagination} data-e2e={`Pagination:${id ?? 'wrapper'}`}>
            <EnumSwitcher
               name={'Pagination'}
               value={value.toString()}
               onChange={handlePageClick}
               options={[firstButton, prevButton, ...buttons, nextButton, lastButton]}
            />

            <SimpleSelect
               switcherProps={{ className: classes.select }}
               value={perPage.toString()}
               onUpdate={handlePerPageChange}
               items={perPageVariants.map(v => ({
                  value: v.toString(),
                  title: v.toString(),
               }))}
            />
         </div>
      );
      // endregion
   },
);
Pagination.displayName = 'Pagination';
