import { Tabs, TabsDirection } from '@yandex-cloud/uikit';
import { TabItem } from '@yandex-data-ui/common';
import React, { ReactElement, ReactNode, useCallback, useEffect, useState } from 'react';

import { classNames } from '../../formatters';

import { SmartTabProps } from './SmartTab';
import classes from './SmartTabs.module.css';

export interface SmartTabsProps {
   /**
    * Дополнительный элемент рядом с вкладками.
    */
   actions?: ReactNode;
   activeTab?: string;
   className?: string;
   /**
    * Расположение вкладок
    *
    * По-умолчанию горизонтальное
    */
   direction?: TabsDirection;

   onSelectTab?(tab: string): void;
}

export const SmartTabs: React.FC<SmartTabsProps> = React.memo(
   ({ actions, activeTab, onSelectTab, children, direction = TabsDirection.Horizontal, className }) => {
      const tabs = (React.Children.toArray(children) as ReactElement<SmartTabProps>[])
         // eslint-disable-next-line no-underscore-dangle
         .filter(node => node.props.__ROLE === 'SmartTab')
         .map(tabNode => {
            const { props } = tabNode;

            return {
               content: props.children,
               disabled: props.disabled,
               id: props.id,
               isEmpty: props.isEmpty,
               title: tabNode,
            } as TabItem<any> & { content: ReactNode; isEmpty: boolean };
         });

      // Либо явно переданный, либо первый не пустой, либо просто первый
      const [currentTab, setCurrentTab] = useState(activeTab ?? tabs?.find(t => !t.isEmpty)?.id ?? tabs?.[0]?.id ?? '');

      useEffect(() => {
         if (activeTab) {
            setCurrentTab(activeTab);
         }
      }, [activeTab]);

      const handleSelectTab = useCallback(
         (tab: string) => {
            setCurrentTab(tab);

            if (onSelectTab) {
               onSelectTab(tab);
            }
         },
         [onSelectTab],
      );

      return (
         <div
            className={classNames(
               {
                  [classes.wrapperHorizontal]: direction === TabsDirection.Horizontal,
                  [classes.wrapperVertical]: direction === TabsDirection.Vertical,
               },
               className,
            )}
         >
            <div className={classes.tabs}>
               <Tabs items={tabs} activeTab={currentTab} direction={direction} onSelectTab={handleSelectTab} />
               {actions}
            </div>

            {direction === TabsDirection.Vertical && <div className={classes.divider} />}

            {tabs
               .filter(tab => tab.id === currentTab)
               .map(tab => (
                  <div key={tab.id} className={classes.content}>
                     {tab.content}
                  </div>
               ))}
         </div>
      );
   },
);

SmartTabs.displayName = 'SmartTabs';
