import { Observable, of } from 'rxjs';
import { map } from 'rxjs/operators';

import { BaseApi, HttpMethod } from '../../http';

interface StaffApiRequest {
   /**
    * Page. Default: 1.
    */
   _page?: number;

   /**
    * Object per page. Default: 50.
    */
   _limit?: number;

   /**
    * Return one object without pagination container. Value: 1.
    */
   _one?: number;

   /**
    * Format response body. Value: 1.
    */
   _pretty?: number;

   /**
    * Comma-separated list of fields to return.
    */
   _fields?: string[];

   /**
    * Render this documentation page. Value: 1.
    */
   _doc?: number;

   /**
    * Comma-separated list of field names. Default: -_meta.modified_at.
    */
   _sort?: string[];

   /**
    * PQL query to execute. See https://github.com/alonho/pql for details.
    */
   _query?: string;

   // <field> Comma-separated list of field values to filter.
}

interface StaffApiResponse<T> {
   result: T[];
   total: number;
   pages: number;
   limit: number;
   page: number;
}

interface StaffGroup {
   id: number;
   url: string;
   name: string;
}

const DEFAULT_STAFF_API_ENDPOINT = 'https://staff-api.yandex-team.ru/v3/';

export class StaffApi extends BaseApi {
   protected getCSRF(): string {
      return '';
   }

   protected handleError(): void {}

   constructor(apiUrl = DEFAULT_STAFF_API_ENDPOINT) {
      super(apiUrl);
   }

   public resolveGroupIds(groupIds: number[]): Observable<StaffGroup[]> {
      if (groupIds.length === 0) {
         return of([]);
      }

      return this.request<StaffApiRequest, void, StaffApiResponse<StaffGroup>>(HttpMethod.GET, 'groups', {
         _limit: 1000,
         _query: `id in [${groupIds.join(',')}] and is_deleted == false`,
         _fields: ['id', 'name', 'url'],
      }).pipe(map(r => r.result));
   }
}
