import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent, useEffect, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { IUserOrGroup, UserOrGroupType } from '../../_models';
import { icuiTimes } from '../../_styles/@fortawesome/ui-libs-icons';
import { classNames } from '../../formatters';
import { useDismounted } from '../../react_hooks';
import { UserName } from '../UserName/UserName';

import { StaffApi } from './StaffApi';
import classes from './UserList.module.css';

function isNumberOrStringOfDigits(text: string | number) {
   return typeof text === 'number' || /^\d+$/.test(text);
}

interface IProps {
   addGroupType?: boolean;
   cls?: string;
   editable: boolean;
   forceAvatars?: boolean;
   resolveGroups?: boolean;
   staffApiEndpoint?: string;
   value: IUserOrGroup[];

   onChange?(e: SyntheticEvent, v: IUserOrGroup[]): void;
}

/**
 * Список логинов пользователей или групп с возможностью удаления.
 */
export const UserList = React.memo(
   ({
      addGroupType = false,
      cls = '',
      editable,
      forceAvatars = false,
      onChange,
      resolveGroups,
      staffApiEndpoint,
      value,
   }: IProps) => {
      if (editable && !onChange) {
         throw new Error('UserList need onChange if editable');
      }

      const dismounted = useDismounted();
      const staffApi = useMemo(() => new StaffApi(staffApiEndpoint), [staffApiEndpoint]);

      // null - признак того, что запись была запрошена, но не найдена
      const [resolvedItems, setResolvedItems] = useState<Record<string, IUserOrGroup | null>>({});

      // handlers
      const remove = (userOrGroup: IUserOrGroup) => (e: SyntheticEvent) => {
         onChange!(
            e,
            value.filter(v => v.id !== userOrGroup.id),
         );
      };

      // Загрузка групп
      useEffect(() => {
         if (resolveGroups && value.some(v => v.type === UserOrGroupType.Group)) {
            // Список id групп с числом в поле id, которые еще не загружены
            const groupIds = value
               .filter(v => v.type === UserOrGroupType.Group && isNumberOrStringOfDigits(v.id))
               .filter(v => !resolveGroups.hasOwnProperty(v.id))
               .map(v => parseInt(v.id, 10));

            staffApi
               .resolveGroupIds(groupIds)
               .pipe(takeUntil(dismounted))
               .subscribe(groups => {
                  setResolvedItems(rg => {
                     groupIds.forEach(groupId => {
                        const group = groups.find(g => g.id === groupId);
                        rg[groupId.toString()] = {
                           id: group?.url ?? groupId.toString(),
                           name: group?.name ?? (
                              <>
                                 {groupId} <span className={classes.notFound}>[not found]</span>
                              </>
                           ),
                           type: UserOrGroupType.Group,
                        };
                     });

                     return { ...rg }; // recreate object to force rerender
                  });
               });
         }
      }, [dismounted, resolveGroups, staffApi, value]);

      // render
      if (value.length === 0) {
         return null;
      }

      const itemClass = classNames(classes.user, {
         [classes.editable]: editable,
      });

      const users = Array.from(value)
         .map(userOrGroup => resolvedItems[userOrGroup.id] ?? userOrGroup)
         .map(userOrGroup => (
            <div key={userOrGroup.id} className={itemClass}>
               <UserName addGroupType={addGroupType} value={userOrGroup} avatar={forceAvatars || undefined} />

               {editable ? (
                  <Button view={'flat'} size={'s'} className={classes.deleteButton} onClick={remove(userOrGroup)}>
                     <FontAwesomeIcon icon={icuiTimes} size={'xs'} />
                  </Button>
               ) : null}
            </div>
         ));

      return <div className={classNames(classes.userList, cls)}>{users}</div>;
   },
);

UserList.displayName = 'UserList';
