import * as React from 'react';

import { EMAIL_HOST, IUserOrGroup, UserOrGroupType, YT_EMAIL_REGEXP } from '../../_models';
import { classNames } from '../../formatters';
// noinspection ES6PreferShortImport
import { AccentFirstLetter } from '../AccentFirstLetter/AccentFirstLetter';
// noinspection ES6PreferShortImport
import { Avatar } from '../Avatar/Avatar';

import { ABC_SCOPES_RESPONSE } from './abcScopes';
import classes from './UserName.module.css';

const scopes = ABC_SCOPES_RESPONSE.results.map(s => s.slug);

const abcServiceWithScope = new RegExp(`^svc_([\\w-]+?)_(${scopes.join('|')})$`);

export function getAbcServiceLink(wikiGroup: string): string {
   const withScopeMatches = wikiGroup.match(abcServiceWithScope);
   if (withScopeMatches) {
      return `https://abc.yandex-team.ru/services/${withScopeMatches[1]}/?scope=${withScopeMatches[2]}`;
   }

   return `https://abc.yandex-team.ru/services/${wikiGroup.slice(4)}/`;
}

export function getStaffDepartmentLink(department: string): string {
   return `https://staff.yandex-team.ru/departments/${department}`;
}

interface Props {
   addGroupType?: boolean;
   avatar?: boolean;
   className?: string;
   hideAvatar?: boolean;
   value: IUserOrGroup;
}

export const UserName = React.memo(
   ({ addGroupType = false, avatar, className = '', hideAvatar = false, value }: Props) => {
      const isPeople = value.type === UserOrGroupType.People;
      const normalizedLogin = YT_EMAIL_REGEXP.test(value.id) ? value.id.replace(EMAIL_HOST, '') : value.id;

      const useAvatar = avatar !== undefined ? avatar : isPeople || value.type === UserOrGroupType.System;

      let { url } = value;
      let groupType: string | null = null;
      if (!url) {
         switch (value.type) {
            case UserOrGroupType.People: {
               url = `https://staff.yandex-team.ru/${normalizedLogin}`;
               break;
            }

            case UserOrGroupType.Group: {
               if (normalizedLogin.startsWith('svc_')) {
                  url = getAbcServiceLink(normalizedLogin);
                  groupType = 'ABC';
               } else if (normalizedLogin.startsWith('yandex_')) {
                  url = getStaffDepartmentLink(normalizedLogin);
                  groupType = 'Staff';
               }
               break;
            }
            default: {
               url = '';
               break;
            }
         }
      }

      const avatarLogin = useAvatar ? normalizedLogin : value.type === UserOrGroupType.Group ? 'service' : null;

      const content = (
         <>
            {hideAvatar ? null : <Avatar login={avatarLogin} />}
            <AccentFirstLetter value={value.name ?? normalizedLogin} blue={value.type === UserOrGroupType.Group} />
            {addGroupType && groupType ? <span className={classes.groupType}>({groupType})</span> : null}
         </>
      );

      return url ? (
         <a href={url} target={'_blank'} className={classNames(classes.userName, className)} rel={'noreferrer'}>
            {content}
         </a>
      ) : (
         <div className={classNames(classes.userName, className)}>{content}</div>
      );
   },
);

UserName.displayName = 'UserName';
