import { IconDefinition } from '@fortawesome/fontawesome-common-types';
import { faCircleInfo, faExclamationTriangle } from '@fortawesome/free-solid-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Card, CardTheme } from '@yandex-cloud/uikit';
import React, { ReactNode } from 'react';

import { classNames } from '../../formatters';

import classes from './WarningPlate.module.css';

export const cardThemes = new Set<CardTheme>(['danger', 'info', 'normal', 'positive', 'warning']);

export type PlateSize = 'n' | 's' | 'xs';

export const plateSizes = new Set<PlateSize>(['n', 's', 'xs']);

function getActualValue<T extends string>(set: Set<T>, value: T, defaultValue: T): T {
   return set.has(value) ? value : defaultValue;
}

const icons: Record<CardTheme, IconDefinition> = {
   danger: faExclamationTriangle,
   info: faCircleInfo,
   normal: faCircleInfo,
   positive: faCircleInfo,
   warning: faExclamationTriangle,
};

export interface WarningPlateProps {
   className?: string;
   filled?: boolean;
   icon?: IconDefinition;
   secondRaw?: ReactNode;
   size?: PlateSize;
   theme?: CardTheme;
}

/**
 * Информационная плашка
 *
 * Уровень (стиль) по-умолчанию - Warning
 */
export const WarningPlate: React.FC<WarningPlateProps> = React.memo(
   ({ children, className, icon, secondRaw, theme = 'warning', size = 'n', filled = true }) => {
      const actualTheme = getActualValue(cardThemes, theme, 'warning');
      const actualSize = getActualValue(plateSizes, size, 'n');

      return (
         <Card
            className={classNames(
               classes.card,
               classes[`theme_${actualTheme}`],
               classes[`size_${actualSize}`],
               className,
            )}
            theme={actualTheme}
            view={filled ? 'filled' : undefined}
         >
            <FontAwesomeIcon
               className={classes.icon}
               fixedWidth={true}
               icon={icon ?? icons[actualTheme] ?? faCircleInfo}
               size={'lg'}
            />
            <div className={classes.firstRow}>{children}</div>
            {secondRaw ? <div className={classes.secondRow}>{secondRaw}</div> : null}
         </Card>
      );
   },
);

WarningPlate.displayName = 'WarningPlate';
