import { logActions } from '../utils';

import { ISuggestBaseEntity } from './models';

export enum ActionType {
   AfterLoading = 'AfterLoading',
   BeforeLoading = 'BeforeLoading',
   Close = 'Close',
   Open = 'Open',
   Select = 'Select',
   SetQuery = 'SetQuery',
}

interface IActionAfterLoading {
   items: ISuggestBaseEntity[];
   type: ActionType.AfterLoading;
}

interface IActionBeforeLoading {
   type: ActionType.BeforeLoading;
}

interface IActionClose {
   type: ActionType.Close;
}

interface IActionOpen {
   type: ActionType.Open;
}

interface IActionSelect {
   query: string;
   selected: ISuggestBaseEntity;
   type: ActionType.Select;
}

interface IActionSetQuery {
   opened?: boolean;
   query: string;
   type: ActionType.SetQuery;
}

export type Action =
   | IActionAfterLoading
   | IActionBeforeLoading
   | IActionClose
   | IActionOpen
   | IActionSelect
   | IActionSetQuery;

export const initialState = {
   isLoading: false,
   items: [] as ISuggestBaseEntity[],
   opened: false,
   query: '',
   selected: null as ISuggestBaseEntity | null,
};

type State = Readonly<typeof initialState>;

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.AfterLoading: {
         return {
            ...state,
            isLoading: false,
            items: action.items,
         };
      }

      case ActionType.BeforeLoading: {
         return {
            ...state,
            isLoading: true,
            items: [],
         };
      }

      case ActionType.Close: {
         return {
            ...state,
            opened: false,
         };
      }

      case ActionType.Open: {
         return {
            ...state,
            items: [],
            opened: true,
         };
      }

      case ActionType.Select: {
         return reducer(
            {
               ...state,
               query: action.query,
               selected: action.selected,
            },
            { type: ActionType.Close },
         );
      }

      case ActionType.SetQuery: {
         return {
            ...state,
            opened: action.opened === undefined ? state.opened : action.opened,
            query: action.query,
            selected: state.query === action.query ? state.selected : null,
         };
      }

      default: {
         return state;
      }
   }
};

export const reducerWithLogs = logActions(reducer);
