// eslint-disable-next-line max-classes-per-file
import { Meta } from '@storybook/react';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useRef, useState } from 'react';
import { Observable, of, throwError, timer } from 'rxjs';
import { mapTo } from 'rxjs/operators';

import { json } from '../formatters';
import { DevJson } from '../small_components';

import { suggestApi } from './api';
import { ISuggestStrategy } from './layers/models';
import { ISuggestBaseEntity, SuggestLayer } from './models';
import { SuggestSelect } from './SuggestSelect';

interface IProps {
   allowCreate?: boolean;
   clearOnClose?: boolean;
   id?: number | string;
   initialQuery?: string;
   layer?: SuggestLayer;
}

function SuggestLayerStory({ allowCreate, layer, initialQuery, id }: IProps) {
   // hooks
   const [query, setQuery] = useState(initialQuery);
   const [value, setValue] = useState<ISuggestBaseEntity | null>(null);

   // handlers
   const updateQuery = useCallback((e: SyntheticEvent | null, q: string) => setQuery(q), []);

   const updateValue = useCallback((e: SyntheticEvent | null, item: ISuggestBaseEntity) => setValue(item), []);

   // render
   return (
      <>
         <SuggestSelect
            allowCreate={allowCreate}
            layer={layer}
            initialQuery={query}
            clearOnSelect={false}
            onQueryUpdate={updateQuery}
            onSelect={updateValue}
            entityId={id}
         />

         <DevJson open={true}>{{ query, value }}</DevJson>
      </>
   );
}

const options: ISuggestBaseEntity[] = [
   { id: 1, title: 'Option 1: Checkbox', layer: SuggestLayer.People, url: '' },
   { id: 2, title: 'Option 2: TextArea', layer: SuggestLayer.People, url: '' },
   { id: 3, title: 'Option 3: Input', layer: SuggestLayer.People, url: '' },
   { id: 4, title: 'Option 4: Select', layer: SuggestLayer.People, url: '' },
];

class CustomStrategy implements ISuggestStrategy {
   public getPlaceholder(): string {
      return 'Custom options';
   }

   public getQueryFromEntity(entity: ISuggestBaseEntity): string {
      return entity ? entity.title : '';
   }

   public load(query: string): Observable<ISuggestBaseEntity[]> {
      const value = options.filter(o => o.title.toLowerCase().includes(query.toLowerCase()));

      return timer(250).pipe(mapTo(value));
   }

   public renderItem(entity: ISuggestBaseEntity): JSX.Element {
      return (
         <div>
            {entity.id}: {entity.title}
         </div>
      );
   }
}

function CustomSuggestStory({ clearOnClose }: IProps) {
   // hooks
   const customStrategy = useRef(new CustomStrategy());
   const [query, setQuery] = useState('Checkbox');
   const [value, setValue] = useState<ISuggestBaseEntity | null>(null);

   // handlers
   const updateQuery = (e: SyntheticEvent | null, q: string) => setQuery(q);

   const updateValue = (e: SyntheticEvent | null, item: ISuggestBaseEntity) => setValue(item);

   const onBlur = (e: React.FocusEvent<HTMLInputElement>) => {
      console.log('blur', e.target.name);
   };

   // render
   return (
      <>
         <pre>{json(options)}</pre>

         <SuggestSelect
            allowEmptyQuery={true}
            strategy={customStrategy.current}
            initialQuery={query}
            clearOnClose={clearOnClose}
            clearOnSelect={false}
            onQueryUpdate={updateQuery}
            onSelect={updateValue}
            name={'customSuggestSelect'}
            onBlur={onBlur}
         />

         <pre>{json({ query, value })}</pre>
      </>
   );
}

const SimpleRequestStory = () => {
   const [data, setData] = useState<any>(null);

   useEffect(() => {
      suggestApi.findPeople('khoden').subscribe(setData);
   }, []);

   return <pre>{json(data)}</pre>;
};

export const People = () => <SuggestLayerStory layer={SuggestLayer.People} />;

export const PeopleWithValue = () => <SuggestLayerStory layer={SuggestLayer.People} initialQuery={'khoden'} />;

export const PeopleAllowCreate = () => <SuggestLayerStory layer={SuggestLayer.People} allowCreate={true} />;

export const Groups = () => <SuggestLayerStory layer={SuggestLayer.Groups} />;

export const PeopleOrGroups = () => <SuggestLayerStory layer={SuggestLayer.PeopleOrGroups} />;

export const TrackerQueues = () => <SuggestLayerStory layer={SuggestLayer.Queues} />;

export const TrackerIssues = () => <SuggestLayerStory layer={SuggestLayer.Issues} />;

export const Services = () => <SuggestLayerStory layer={SuggestLayer.Services} id={3494} />;

export const Unknown = () => <SuggestLayerStory layer={'fake-suggest-layer' as SuggestLayer} />;

export const Custom = () => <CustomSuggestStory />;

export const CustomClearOnPopupClose = () => <CustomSuggestStory clearOnClose={true} />;

export const AvoidPreflightRequests = () => <SimpleRequestStory />;

class ErrorStrategy implements ISuggestStrategy {
   public getPlaceholder(): string {
      return 'Type xxx to fail';
   }

   public getQueryFromEntity(): string {
      return '';
   }

   public load(query: string): Observable<ISuggestBaseEntity[]> {
      if (query === 'xxx') {
         return throwError(new Error(`Stub network error for ${query}`));
      }

      return of([
         {
            id: query,
            layer: 'custom' as SuggestLayer,
            title: query,
            url: query,
         },
      ]);
   }

   public renderItem(entity: ISuggestBaseEntity): JSX.Element {
      return <div>{entity.id}</div>;
   }
}

export const LoadError = () => {
   // hooks
   const customStrategy = useRef(new ErrorStrategy());
   const [, setValue] = useState<ISuggestBaseEntity | null>(null);

   // handlers
   const updateValue = (e: SyntheticEvent | null, item: ISuggestBaseEntity) => setValue(item);
   const handleError = useCallback((error: any) => {
      console.log(`handleError`, error);
   }, []);

   // render
   return (
      <SuggestSelect
         allowEmptyQuery={true}
         strategy={customStrategy.current}
         onSelect={updateValue}
         name={'errorSuggestSelect'}
         onError={handleError}
      />
   );
};

export default {
   title: 'suggestions/SuggestSelect',
} as Meta;
