import { Popup, TextInput } from '@yandex-cloud/uikit';
import * as React from 'react';
import { MutableRefObject, SyntheticEvent, useCallback, useEffect, useMemo, useReducer, useRef } from 'react';
import { Subscription } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import { isEmpty } from '../helpers';
import { useDismounted, useDropdownDimensions } from '../react_hooks';
import { Loader } from '../small_components';
import { toasts } from '../toasts';

import { suggestApi } from './api';
import { ISuggestStrategy } from './layers/models';
import { getStrategy } from './layers/strategies';
import { EntityId, ISuggestBaseEntity, SuggestLayer } from './models';
import { EmptySuggestItem, SuggestItem } from './SuggestItem';
import styles from './SuggestSelect.module.css';
import { Action, ActionType, initialState, reducer } from './SuggestSelect.state';

interface IProps {
   allowCreate?: boolean;
   allowEmptyQuery?: boolean;
   autofocus?: boolean;
   clearOnClose?: boolean;
   clearOnSelect?: boolean;
   cls?: string;
   disabled?: boolean;
   entityId?: EntityId;
   hasClear?: boolean;
   initialQuery?: string;
   layer?: SuggestLayer | string;
   name?: string;
   qa?: string;
   resolveItemOnInit?: boolean;
   strategy?: ISuggestStrategy;

   onBlur?(e: React.FocusEvent<HTMLInputElement>): void;

   onError?(error: any): void;

   onQueryUpdate?(e: SyntheticEvent | null, q: string): void;

   onSelect(e: SyntheticEvent | null, item: ISuggestBaseEntity | null): void;
}

type QueryCache = Map<string, ISuggestBaseEntity[]>;

const getLoader =
   (
      allowEmptyQuery: boolean,
      strategy: ISuggestStrategy,
      cache: QueryCache,
      dispatch: (action: Action) => void,
      requestRef: MutableRefObject<Subscription | null>,
      onError: ((error: any) => void) | undefined,
   ) =>
   (q: string, opened: boolean) => {
      if (!opened) {
         return;
      }

      if (!allowEmptyQuery && isEmpty(q)) {
         dispatch({ type: ActionType.AfterLoading, items: [] });

         return;
      }

      if (cache.has(q)) {
         dispatch({ type: ActionType.AfterLoading, items: cache.get(q)! });

         return;
      }

      dispatch({ type: ActionType.BeforeLoading });

      if (requestRef.current) {
         requestRef.current.unsubscribe();
      }

      requestRef.current = strategy.load(q).subscribe(
         items => {
            cache.set(q, items);
            dispatch({ type: ActionType.AfterLoading, items });
         },
         (e: any) => {
            toasts.apiError('Load suggestions', e);
            dispatch({ type: ActionType.AfterLoading, items: [] });
            onError?.(e);
         },
      );
   };

export const SuggestSelect: React.FC<IProps> = React.memo(
   ({
      allowCreate = false,
      allowEmptyQuery = false,
      autofocus = false,
      clearOnClose = false,
      clearOnSelect = true,
      cls = '',
      disabled = false,
      entityId,
      hasClear = true,
      initialQuery = '',
      layer,
      name,
      onBlur,
      onError,
      onQueryUpdate,
      onSelect,
      qa,
      resolveItemOnInit = false,
      strategy: propStrategy,
   }) => {
      if (!propStrategy && !layer) {
         throw new Error('You should define layer or strategy prop for SuggestSelect');
      }

      // hooks
      const strategy = useMemo(() => propStrategy || getStrategy(layer!), [propStrategy, layer]);
      const cache = useMemo(() => new Map() as QueryCache, []);
      const listRef = useRef<HTMLDivElement>(null);
      const switcherRef = useRef<HTMLInputElement>(null);
      const dismounted = useDismounted();

      const [, maxHeight] = useDropdownDimensions(switcherRef as any);

      const [{ isLoading, items: suggestions, opened, query, selected }, dispatch] = useReducer(reducer, {
         ...initialState,
         query: initialQuery,
      });
      const requestRef = useRef<Subscription | null>(null);

      const loader = useMemo(
         () => getLoader(allowEmptyQuery, strategy, cache, dispatch, requestRef, onError),
         [allowEmptyQuery, cache, strategy, onError],
      );

      // effects
      useEffect(
         // Загрузка данных при изменении запроса или открытии
         () => {
            if (!opened) {
               return;
            }

            if (onQueryUpdate) {
               onQueryUpdate(null, query);
            }

            if (isEmpty(query)) {
               onSelect(null, null);
            }

            loader(query, opened);
         },
         [query, opened, loader, onQueryUpdate, onSelect],
      );

      useEffect(
         // Изменение запроса, если внешний изменился
         () => {
            if (initialQuery) {
               dispatch({ type: ActionType.SetQuery, query: initialQuery });

               return;
            }

            if (resolveItemOnInit && entityId !== undefined) {
               // Загрузка сущности при инициализации
               strategy
                  .load(String(entityId))
                  .pipe(takeUntil(dismounted))
                  .subscribe(entities => {
                     const entity = entities.find(x => x.id === entityId);

                     if (entity) {
                        dispatch({ type: ActionType.SetQuery, query: strategy.getQueryFromEntity(entity) });
                     }
                  });
            }
         },
         [initialQuery, entityId, resolveItemOnInit, strategy, dismounted],
      );

      useEffect(
         () =>
            // Отмена текущего запроса при смерти
            () => {
               if (requestRef.current) {
                  requestRef.current.unsubscribe();
               }
            },
         [],
      );

      // handlers
      const open = useCallback(() => dispatch({ type: ActionType.Open }), []);

      const close = useCallback(() => dispatch({ type: ActionType.Close }), []);

      const updateQuery = useCallback((v: string) => {
         dispatch({ type: ActionType.SetQuery, query: v, opened: true });
      }, []);

      const focusInput = useCallback(() => {
         if (switcherRef.current) {
            switcherRef.current.focus();
         }
      }, []);

      const onSuggestSelect = useCallback(
         (e: SyntheticEvent, entity: ISuggestBaseEntity) => {
            suggestApi.reportClick(entity);
            dispatch({
               query: clearOnSelect ? '' : strategy.getQueryFromEntity(entity),
               selected: entity,
               type: ActionType.Select,
            });
            onSelect(e, entity);
         },
         [clearOnSelect, onSelect, strategy],
      );

      const handleClose = useCallback(() => {
         if (clearOnClose) {
            const q = selected && strategy.getQueryFromEntity(selected) === query ? query : '';

            dispatch({ query: q, opened: false, type: ActionType.SetQuery });
         } else {
            close();
         }
      }, [clearOnClose, close, query, selected, strategy]);

      const onKeyDown = useCallback(
         (e: React.KeyboardEvent<HTMLElement>) => {
            switch (e.key) {
               case 'Tab':
               case 'ArrowDown':
                  if (listRef.current && listRef.current.firstChild) {
                     e.preventDefault();
                     (listRef.current.firstChild as HTMLDivElement).focus();
                  }
                  break;

               case 'ArrowUp':
                  if (listRef.current && listRef.current.lastChild) {
                     e.preventDefault();
                     (listRef.current.lastChild as HTMLDivElement).focus();
                  }
                  break;

               case 'Enter':
                  if (allowCreate && !isEmpty(query) && strategy.getEntityFromQuery) {
                     const entity = strategy.getEntityFromQuery(query, suggestions);
                     onSuggestSelect(e, entity);
                  }
                  break;

               default:
                  break;
            }
         },
         [allowCreate, query, strategy, suggestions, onSuggestSelect],
      );

      // render

      return (
         <>
            <TextInput
               autoFocus={autofocus}
               className={cls}
               controlProps={{ autoComplete: 'off' }}
               controlRef={switcherRef}
               disabled={disabled}
               hasClear={hasClear}
               id={name}
               name={name}
               onBlur={onBlur}
               onFocus={open}
               onKeyDown={onKeyDown}
               onUpdate={updateQuery}
               placeholder={strategy.getPlaceholder()}
               qa={qa}
               value={query}
            />
            <Popup
               anchorRef={switcherRef}
               className={styles.popup}
               onClose={handleClose}
               open={opened}
               placement={['bottom-start']}
            >
               {isLoading ? (
                  <EmptySuggestItem>
                     <Loader />
                  </EmptySuggestItem>
               ) : suggestions.length === 0 ? (
                  <EmptySuggestItem>No suggestions</EmptySuggestItem>
               ) : (
                  <div ref={listRef} style={{ maxHeight, overflowY: 'auto' }}>
                     {suggestions.map(entity => (
                        <SuggestItem
                           key={entity.id}
                           item={entity}
                           onSelect={onSuggestSelect}
                           focusInput={focusInput}
                           avoidAddingPadding={strategy.avoidAddingPaddings}
                        >
                           {strategy.renderItem(entity, query)}
                        </SuggestItem>
                     ))}
                  </div>
               )}
            </Popup>
         </>
      );
   },
);

SuggestSelect.displayName = 'SuggestSelect';
