import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';

import { ApiCache, BaseApi, HttpMethod, IApiError, PersistentStore } from '../http';

import { IStaffGroup } from './layers/groups/models';
import { IIssue } from './layers/issues/models';
import { IStaffPerson } from './layers/people/models';
import { StaffPersonOrGroup } from './layers/peopleOrGroup/models';
import { IQueue } from './layers/queues/models';
import { IAbcService } from './layers/services/models';
import { ISuggestBaseEntity, ISuggestBaseParams, ISuggestResponse, SuggestLanguage, SuggestLayer } from './models';

/**
 * Урезанная версия, реализует только срез функционала для реализации подсказок при поиске сотрудника
 */
class SuggestApi extends BaseApi {
   private cache = new ApiCache(new PersistentStore(3600, 'SuggestApi'));

   private groupParams = {
      'groups.per_page': 10,
   };

   private issueParams = {
      'issues.per_page': 10,
   };

   private language: SuggestLanguage = 'en';

   private peopleParams = {
      'people.per_page': 10,
      'people.query': 'i_is_dismissed:0',
   };

   private queueParams = {
      'queues.per_page': 10,
   };

   private serviceParams = {
      'services.per_page': 10,
   };

   private version = 2;

   public findAbcServices(query: string): Observable<IAbcService[]> {
      return this.cache.request(`services/${query}`, () =>
         this.find(SuggestLayer.Services, query, this.serviceParams).pipe(
            map(resp => this.getResult<IAbcService>(resp, SuggestLayer.Services)),
         ),
      );
   }

   public findPeople(query: string): Observable<IStaffPerson[]> {
      return this.cache.request(`people/${query}`, () =>
         this.find(SuggestLayer.People, query, this.peopleParams).pipe(
            map(resp => this.getResult<IStaffPerson>(resp, SuggestLayer.People)),
         ),
      );
   }

   public findGroups(query: string): Observable<IStaffGroup[]> {
      return this.cache.request(`groups/${query}`, () =>
         this.find(SuggestLayer.Groups, query, this.groupParams).pipe(
            map(resp => this.getResult<IStaffGroup>(resp, SuggestLayer.Groups)),
         ),
      );
   }

   public findQueues(query: string): Observable<IQueue[]> {
      return this.find(SuggestLayer.Queues, query, this.queueParams).pipe(
         map((resp: any) => resp[SuggestLayer.Queues].result),
      );
   }

   public findIssues(query: string): Observable<IIssue[]> {
      return this.find(SuggestLayer.Issues, query, this.issueParams).pipe(
         map((resp: any) => resp[SuggestLayer.Issues].result),
      );
   }

   public findPeopleOrGroups(query: string): Observable<StaffPersonOrGroup[]> {
      const params = {
         ...this.groupParams,
         ...this.peopleParams,
      };

      return this.cache.request(`peopleOrGroups/${query}`, () =>
         this.find(SuggestLayer.PeopleOrGroups, query, params).pipe(
            map(resp => [
               ...this.getResult<IStaffPerson>(resp, SuggestLayer.People),
               ...this.getResult<IStaffGroup>(resp, SuggestLayer.Groups),
            ]),
         ),
      );
   }

   public reportClick(entity: ISuggestBaseEntity): void {
      if (entity.click_urls) {
         for (const url of entity.click_urls) {
            this.request(HttpMethod.GET, url).subscribe(
               () => undefined,
               () => undefined,
            );
         }
      }
   }

   protected getCSRF(): string {
      return ''; // нерелевантно
   }

   // eslint-disable-next-line @typescript-eslint/no-unused-vars
   protected handleError(resp: Response, error: IApiError): void {
      // нерелевантно
   }

   private find(layer: SuggestLayer, text: string, params?: object): Observable<ISuggestResponse> {
      const allParams = {
         // Базовые параметры запроса
         ...this.getDefaultParams(layer, text),
         ...params,
      };

      return this.request<any, void, ISuggestResponse>(HttpMethod.GET, '', allParams);
   }

   private getDefaultParams(layer: SuggestLayer, query: string): ISuggestBaseParams {
      return {
         language: this.language,
         layers: layer,
         text: query,
         version: this.version,
      };
   }

   private getResult<T extends ISuggestBaseEntity>(resp: ISuggestResponse, layer: SuggestLayer): T[] {
      return (resp[layer]?.result as T[]) || [];
   }
}

export const suggestApi = new SuggestApi('//search.yandex-team.ru/suggest/');
