/**
 * Производит выборку из большого массива
 *
 * Иногда нужно получать suggestions локально, из большого массива возможных значений.
 * Функциональный подход с filter, map, reduce приведёт к полной итерации всего массива.
 * Хотелось бы этого избежать, если нам нужно получить limit первых подходящих элементов.
 * Поэтому написано в классическом императивном стиле без заморочек с трансдьюсерами или генераторами.
 *
 * @param items Исходный массив
 * @param filter Предикат выбора
 * @param limit Максимальное число элементов в результате (при достижении итерирование прекращается)
 * @param map  Функция преобразования элемента массива в пункт suggestions
 */
import { sortHandler } from '../helpers';

export function getSuggestions<I, O>(
   items: I[],
   filter: (item: I) => boolean,
   limit: number,
   map: (item: I) => O,
): O[] {
   const result: O[] = [];

   for (let i = 0, l = items.length; i < l; i += 1) {
      const item = items[i];
      if (filter(item)) {
         result.push(map(item));
      }

      if (result.length >= limit) {
         break;
      }
   }

   return result;
}

export interface IItemWithWeight<O> {
   item: O;
   weight: number;
}

export function getWeightedSuggestions<I, O>(
   items: I[],
   map: (item: I) => O,
   limit: number,
   ...filters: ((item: I) => number | null)[]
): O[] {
   const result: IItemWithWeight<O>[] = [];

   for (let i = 0, l = items.length; i < l; i += 1) {
      const item = items[i];

      for (const filter of filters) {
         const weight = filter(item);
         if (weight !== null) {
            result.push({ item: map(item), weight });
            break;
         }
      }
   }

   result.sort((a, b) => sortHandler(a.weight, b.weight));

   return result.slice(0, limit).map(i => i.item);
}
