// eslint-disable-next-line max-classes-per-file
import { autobind, bindAndMemoize } from './decorators';

describe('decorators', () => {
   describe('@autobind', () => {
      class A {
         public getXReactWay = () => this.x;

         // tslint:disable-next-line:member-ordering
         private x = 1;

         @autobind
         public getXBound() {
            return this.x;
         }

         public getXUnbound() {
            return this.x;
         }
      }

      let a: A;
      beforeEach(() => {
         a = new A();
      });

      it('should bind with reactWay', () => {
         const method = a.getXReactWay;

         expect(method()).toBe(1);
      });

      it('should be same function with reactWay', () => {
         const method1 = a.getXReactWay;
         const method2 = a.getXReactWay;

         expect(method1 === method2).toBe(true);
      });

      it('should bind with @autobind', () => {
         const method = a.getXBound;

         expect(method()).toBe(1);
      });

      it('should be same function with @autobind', () => {
         const method1 = a.getXBound;
         const method2 = a.getXBound;

         expect(method1 === method2).toBe(true);
      });

      it('should failed without decorator', () => {
         const method = a.getXUnbound;

         expect(() => method()).toThrow(TypeError);
      });

      it('should be same without decorator', () => {
         const method1 = a.getXUnbound;
         const method2 = a.getXUnbound;

         expect(method1 === method2).toBe(true);
      });
   });

   describe('@bindAndMemoize', () => {
      class A {
         constructor(private logger: (...args: string[]) => void) {}

         public onClick(a: string) {
            return (b: string) => {
               this.logger(a, b);
            };
         }

         @bindAndMemoize
         public onClickMemoized(a: string) {
            return (b: string) => {
               this.logger(a, b);
            };
         }
      }

      let obj: A;

      function log(a: string, b: string) {
         console.log(a, b);
      }

      beforeEach(() => {
         obj = new A(log);
      });

      it('should not memoize without @bindAndMemoize', () => {
         const method1 = obj.onClick('a');
         const method2 = obj.onClick('a');

         expect(method1 === method2).toBe(false);
      });

      it('should memoize with @bindAndMemoize', () => {
         const method1 = obj.onClickMemoized('a');
         const method2 = obj.onClickMemoized('a');

         expect(method1 === method2).toBe(true);
      });

      it('should not memoize with @bindAndMemoize for different values', () => {
         const method1 = obj.onClickMemoized('a');
         const method2 = obj.onClickMemoized('b');

         expect(method1 === method2).toBe(false);
      });
   });
});
